/**
 * -------------------------------------------------------
 * JavaScript add-on methods for Number and String objects
 * -------------------------------------------------------
 */

/* jshint unused:true, undef:true */
/* globals mixin, Object, Date */

// Wrap in function.
(function() {

/**
 * Function instanceOf(obj,type)
 * 
 * <p>Checks if an object "obj" is instance of a type "type", case sensitive.
 * 
 * @param obj Some object to test.
 * @param {String} type The type to check for.
 * 
 * @returns true if type is the same or an instance of the type, false otherwise.
 */
Object.instanceOf=function(obj,type)
  {
  /* jshint evil:true */
  return eval('typeof(arguments[0])=="'+type.toLowerCase()+'"||arguments[0] instanceof '+type);
  };

// Polyfill Date.now() function, just in case...
// Normally most browser have it.
if ( !Date.now )
  Date.now=function now()
    {
    return new Date().getTime();
    };
  
///
/// --- Number ---
///
  
mixin(Number.prototype,
  {
  /**
   * Function radix(r,n,c)
   * 
   * <p>Transform the number object to string in accordance with a scale of notation If
   * it is necessary the numeric string will aligned to right and filled by '0' character, by default.
   * 
   * @type Function
   * 
   * @param {Number} r  Radix of scale of notation (it have to be greater or equal 2 and below or equal 36).
   * @param {Number} n  Width of numeric string.
   * @param {String} c  Padding character (by default, '0').
   *
   * @return {String} Numeric string.
   */
  radix: function(r,n,c)
    {
    return this.toString(r).padding(-n,c);
    },

  /**
   * Function bin(number,string)
   * 
   * <p>Transform the number object to string of binary presentation.
   * 
   * @type Function
   * 
   * @param n  Width of numeric string.
   * @param c  Padding character (by default, '0').
   * 
   * @return {String} Numeric string.
   */
  bin: function(n,c)
    {
    return this.radix(0x02,n,c);
    },

  /**
   * oct(number, string)
   * 
   * <p>Transform the number object to string of octal presentation.
   * 
   * @type Function
   * 
   * @param n  Width of numeric string.
   * @param c  Padding character (by default, '0').
   * 
   * @return {String} Numeric string.
   */
  oct: function(n,c)
    {
    return this.radix(0x08,n,c);
    },

  /**
   * Function dec(number, string)
   * 
   * <p>Transform the number object to string of decimal presentation.
   * 
   * @type Function
   * 
   * @param n  Width of numeric string
   * @param c  Padding character (by default, '0')
   * 
   * @return {String} Numeric string
   */
  dec: function(n,c)
    {
    return this.radix(0x0A,n,c);
    },

  /**
   * Function hexl(number, string)
   * 
   * <p>Transform the number object to string of hexadecimal presentation in lower-case of major
   * characters (0-9 and a-f).
   * 
   * @type Function
   * 
   * @param {Number} n  Width of numeric string.
   * @param {String} c  Padding character (by default, '0').
   * 
   * @return {String} Numeric string
   */
  hexl: function(n,c)
    {
    return this.radix(0x10,n,c);
    },

  /**
   * Function hex(number, string)
   * 
   * <p>Transform the number object to string of hexadecimal presentation in upper-case of major
   * characters (0-9 and A-F).
   * 
   * @type Function
   * 
   * @param n  Width of numeric string.
   * @param c  Padding character (by default, '0').
   * 
   * @return {String} Numeric string.
   */
  hex: function(n,c)
    {
    return this.hexl(n,c).toUpperCase();
    }
  });

///
/// --- String ---
///

/**
 * Function String._hashCode(string)
 * 
 * <p>The hashCode algorithm for a String like in Java.
 * 
 * <p>Returns a hash code for this string. The hash code for a <code>String</code> object is computed as <blockquote>
 * 
 * <pre>
 * s[0]*31&circ;(n-1) + s[1]*31&circ;(n-2) + ... + s[n-1]
 * </pre>
 * 
 * </blockquote> using <code>int</code> arithmetic, where <code>s[i]</code> is the <i>i</i>th character of the
 * string, <code>n</code> is the length of the string, and <code>^</code> indicates exponentiation. (The hash value
 * of the empty string is zero.)
 * 
 * @static
 *
 * @type Function
 * 
 * @param {String} str The string to hash.
 * 
 * @returns {Number} a hash code value for this String object.
 * 
 * @memberOf String
 */
String._hashCode=function(str)
  {
  var h=str.__hash||0,ii,cc;
  if ( h<=0 )
    {
    h=0;
    for ( ii=0,cc=str.length; ii<cc; ++ii )
      h=(31*h+str.charCodeAt(ii))&0xFFFFFFFF;
    
    str.__hash=h;
    }
  
  return h;
  };
  
// "sprintf" regular expression.
var SRE=/%%|%(\d+[\$#])?([+-])?('.|0| )?(\d*)(\.\d*)?([bcdfosuxX])/g,
    PRE=/^(?:([a-z]+):(?:([a-z]*):)?\/\/)?(?:([^:@]*)(?::([^:@]*))?@)?((?:[a-z0-9_-]+\.)+[a-z]{2,}|localhost|(?:(?:[01]?\d\d?|2[0-4]\d|25[0-5])\.){3}(?:(?:[01]?\d\d?|2[0-4]\d|25[0-5])))(?::(\d+))?(?:([^:\?\#]+))?(?:\?([^\#]+))?(?:\#([^\s]+))?$/i;

mixin(String.prototype,
  {
  /**
   * Property __hash
   * 
   * <p>The hash value of a string, defined if computed, otherwise undefined.
   * 
   * @type Number
   */
  __hash: 0,
  
  /**
   * Function hashCode(string)
   * 
   * <p>The hashCode algorithm for a String like in Java.
   * 
   * <p>Returns a hash code for this string. The hash code for a <code>String</code> object is computed as <blockquote>
   * 
   * <pre>
   * s[0]*31&circ;(n-1) + s[1]*31&circ;(n-2) + ... + s[n-1]
   * </pre>
   * 
   * </blockquote> using <code>int</code> arithmetic, where <code>s[i]</code> is the <i>i</i>th character of the
   * string, <code>n</code> is the length of the string, and <code>^</code> indicates exponentiation. (The hash value
   * of the empty string is zero.)
   *
   * @type Function
   *
   * @returns {Number} a hash code value for this String object.
   */
  hashCode: function()
    {
    return String._hashCode(this);
    },
  
  /**
   * Function endsWith(s)
   * 
   * <p>Function returns true if the string ends with the specified String.
   * 
   * @type Function
   *
   * @param {String} s ending part String.
   * @returns {Boolean} true if the ending part of the string matches, false otherwise.content
   */
  endsWith: function(s)
    {
    var e=s.length,l=this.length;
    return e<l && this.substring(l-e)===s;
    },
  
  /**
   * Function startsWith(s)
   * 
   * <p>Function returns true if the string starts with the specified String.
   * 
   * @type Function
   * 
   * @param {String} s ending part String.
   * 
   * @returns {Boolean} true if the ending part of the string matches, false otherwise.content
   */
  startsWith: function(s)
    {
    var len=s.length;
    return this.length>=len && this.substr(0,len)===s;
    },
  
  /**
   * Function htmlize()
   * 
   * <p>Modifies the string to present within HTML page.
   * 
   * @type Function
   * 
   * @return {String} The string in HTML syntax.
   */
  htmlize: function()
    {
    return this.replaceAll('&','&amp;')
               .replaceAll('"','&quot;')
               .replaceAll('<','&lt;')
               .replaceAll('>','&gt;')
               .replaceAll('\n\n','<br>&nbsp;<br>')
               .replaceAll('\n','<br>');
    },
 
  /**
   * Function camelize()
   * 
   * <p>Camel case converts the name, e.g. "font-family" will become "fontFamily".
   * 
   * @type Function
   * 
   * @return {String} The string converted to camel case.
   */
  camelize: function()
    {
    return this.replace(/-([a-z])/g,function(g)
      {
      return g[1].toUpperCase();
      });
    },
  
  /**
   * Function trim()
   * 
   * <p>Transform the string object removing leading and trailing white spaces.
   * 
   * @type Function
   * 
   * @return {String} The trimmed string.
   */
  trim: function()
    {
    return this.replace(/(^\s*)|(\s*$)/g,"");
    },
  
  /**
   * Function trimLeft()
   * 
   * <p>Transform the string object removing leading white spaces.
   * 
   * @type Function
   * 
   * @return {String}  The trimmed string.
   */
  trimLeft: function()
    {
    return this.replace(/(^\s*)/,"");
    },
  
  /**
   * Function trimRight()
   * 
   * <p>Transform the string object removing trailing white spaces.
   * 
   * @type Function
   * 
   * @return {String}  The trimmed string.
   */
  trimRight: function()
    {
    return this.replace(/(\s*$)/g,"");
    },
  
  /**
   * Function dup()
   * 
   * <p>Transform the string object duplicating the string.
   * 
   * @type Function
   * 
   * @return {String} The duplicated string.
   */
  dup: function()
    {
    var val=this.valueOf();
    return val+val;
    },
  
  /**
   * Function repeat(number)
   * 
   * <p>Transform the string object multiplying the string.
   * 
   * @type Function
   * 
   * @param n  Amount of repeating.
   * 
   * @return {String} The repeated string.
   */
  repeat: function(n)
    {
    return !n || n<=0 || !this.length? '': Array(n+1).join(this.valueOf());
    },
  
  /**
   * Function padding(number,string)
   * 
   * <p>Transform the string object to string of the actual width filling by the padding
   * character (by default ' '). Negative value of width means left padding, and positive value means right one.
   * 
   * @type Function
   * 
   * @param n  Width of string.
   * @param c  Padding character (by default, ' ').
   * 
   * @return {String}  The padded string.
   */
  padding: function(n,c)
    {
    var val=this.valueOf(),pad;
    if ( !n )
      return val;
    
    pad=String(c||' ').charAt(0).repeat(Math.abs(n)-this.length);
    return n<0? pad+val: val+pad;
    },
  
  /**
   * Function padLeft(number,string)
   * 
   * <p>Wrapper for object.padding Transform the string object to string of the actual width
   * adding the leading padding character (by default ' ').
   * 
   * @type Function
   * 
   * @param n  Width of string.
   * @param c  Padding character.
   * 
   * @return {String} The padded string.
   */
  padLeft: function(n,c)
    {
    return this.padding(-Math.abs(n),c);
    },
  
  /**
   * Function alignRight(number,string)
   * 
   * <p>Wrapper for object.padding Synonym for object.padLeft
   * 
   * @type Function
   * 
   * @param number Width of string.
   * @param string Padding character.
   * 
   * @return {String} Aligned string.
   */
  alignRight: String.prototype.padLeft,
  
  /**
   * Function padRight(number,string)
   * 
   * <p>Wrapper for object.padding Transform the string object to string of the actual width
   * adding the trailing padding character (by default ' ')
   * 
   * @type Function
   * 
   * @param n  Width of string.
   * @param c  Padding character.
   * 
   * @return {String} Padded string.
   */
  padRight: function(n,c)
    {
    return this.padding(Math.abs(n),c);
    },
  
  /**
   * Function alignLeft(number, string)
   * 
   * <p>Wrapper for object.padding Synonym for object.padRight
   * 
   * @type Function
   * 
   * @param number Width of string.
   * @param string Padding character.
   * 
   * @return {String} Aligned string.
   */
  alignLeft: String.prototype.padRight,
  
  /**
   * Function format(...args)
   * 
   * <p>Replaces {0}, {1}, etc, with the arguments. An argument can only be defined ONCE.
   * 
   * @type Function
   */
  format: function()
    {
    for ( var s=this,ii=0,d,r;; )
      {
      d=s.indexOf(r='{'+ii+'}');
      if ( d<0 )
        return s;
      
      s=s.replace(r,arguments[ii++]);
      }
    },
 
  /**
   * Function sprintf(format, argument_list)
   *
   * <p>The string function like one in C/C++, PHP, Perl.
   * Each conversion specification is defined as below:
   * 
   * <pre>
   * %[index][alignment][padding][width][precision]type
   *
   * index        An optional index specifier that changes the order of the 
   *              arguments in the list to be displayed.
   * alignment    An optional alignment specifier that says if the result should be 
   *              left-justified or right-justified. The default is 
   *              right-justified; a "-" character here will make it left-justified.
   * padding      An optional padding specifier that says what character will be 
   *              used for padding the results to the right string size. This may 
   *              be a space character or a "0" (zero character). The default is to 
   *              pad with spaces. An alternate padding character can be specified 
   *              by prefixing it with a single quote ('). See the examples below.
   * width        An optional number, a width specifier that says how many 
   *              characters (minimum) this conversion should result in.
   * precision    An optional precision specifier that says how many decimal digits 
   *              should be displayed for floating-point numbers. This option has 
   *              no effect for other types than float.
   * type         A type specifier that says what type the argument data should be 
   *              treated as. Possible types:
   *
   * % - a literal percent character. No argument is required.  
   * b - the argument is treated as an integer, and presented as a binary number.
   * c - the argument is treated as an integer, and presented as the character 
   *      with that ASCII value.
   * d - the argument is treated as an integer, and presented as a decimal number.
   * u - the same as "d".
   * f - the argument is treated as a float, and presented as a floating-point.
   * o - the argument is treated as an integer, and presented as an octal number.
   * s - the argument is treated as and presented as a string.
   * x - the argument is treated as an integer and presented as a hexadecimal 
   *       number (with lower case letters).
   * X - the argument is treated as an integer and presented as a hexadecimal 
   *       number (with upper case letters).
   * </pre>
   * 
   * @type Function
   * 
   * @return {String}
   */
  sprintf: function()
    {
    var args=arguments,index=0,x,ins,fn=null;
    
    /* The callback function accepts the following properties
     *      x.index contains the substring position found at the origin string
     *      x[0] contains the found substring
     *      x[1] contains the index specifier (as \d+\$ or \d+#)
     *      x[2] contains the alignment specifier ("+" or "-" or empty)
     *      x[3] contains the padding specifier (space char, "0" or defined as '.)
     *      x[4] contains the width specifier (as \d*)
     *      x[5] contains the floating-point precision specifier (as \.\d*)
     *      x[6] contains the type specifier (as [bcdfosuxX]) */
    return this.replace(SRE,function()
      {
      if ( arguments[0]=="%%" )
        return "%";
      
      x=[];
      for ( var i=0; i<arguments.length; ++i )
        x[i]=arguments[i]===undefined? "": arguments[i];
      
      ins=x[1]? args[x[1].substring(0,x[1].length-1)-1]: args[index];
      ++index;
      switch(x[6])
        {
        case "b":
          ins=Number(ins);
          fn=Number.prototype.bin;
          break;
          
        case "c":
          ins=String.fromCharCode(ins);
          fn=String.prototype.padding;
          break;
          
        case "d":
        case "u":
          ins=Number(ins);
          fn=Number.prototype.dec;
          break;
          
        case "f":
          ins=Number(ins);
          fn=String.prototype.padding;
          if ( x[5] )
            ins=ins.toFixed(x[5].substr(1));
          else if ( x[4] )
            ins=ins.toExponential(x[4]);
          else
            ins=ins.toExponential();
          
          x[2]=x[2]=="-"? "+": "-"; // Invert sign because this is not number but string
          break;
          
        case "o":
          ins=Number(ins);
          fn=Number.prototype.oct;
          break;
          
        case "s":
          ins=String(ins);
          fn=String.prototype.padding;
          break;
          
        case "x":
          ins=Number(ins);
          fn=Number.prototype.hexl;
          break;
          
        case "X":
          ins=Number(ins);
          fn=Number.prototype.hex;
          break;
        }
 
      return fn.call(ins,x[2]+x[4],x[3].substr(x[3].length-1)||" ");
      });
    },
  
  /**
   * Function compile()
   *
   * <p>This string function compiles the formatting string to the internal function 
   * to accelerate an execution a formatting within loops. 
   *
   * @example
   * // Standard usage of the sprintf method
   * var s = '';
   * for (var p in obj) {
   *     s += '%s = %s'.sprintf(p, obj[p]);
   * }
   *
   * // The more speed usage of the sprintf method
   * var sprintf = '%s = %s'.compile();
   * var s = '';
   * for (var p in obj) {
   *     s += sprintf(p, obj[p]);
   * }
   *
   * @see         String.prototype.sprintf()
   * 
   * @type Function
   * 
   * @return {String}
   */
  compile: function()
    {
    var index=0,x,ins,fn=null;
    
    /*
     * The callback function accepts the following properties
     *      x.index contains the substring position found at the origin string
     *      x[0] contains the found substring
     *      x[1] contains the index specifier (as \d+\$ or \d+#)
     *      x[2] contains the alignment specifier ("+" or "-" or empty)
     *      x[3] contains the padding specifier (space char, "0" or defined as '.)
     *      x[4] contains the width specifier (as \d*)
     *      x[5] contains the floating-point precision specifier (as \.\d*)
     *      x[6] contains the type specifier (as [bcdfosuxX])
     */
    
    var result=this.replace(SRE,function()
      {
      if ( arguments[0]=="%%" )
        return "%";
      
      x=[];
      for ( var i=0; i<arguments.length; ++i )
        x[i]=arguments[i]===undefined? "": arguments[i];
 
      ins='arguments['+(x[1]? x[1].substring(0,x[1].length-1)-1: index)+']';
      ++index;
      switch(x[6])
        {
        case "b":
          ins='Number('+ins+')';
          fn='Number.prototype.bin';
          break;
          
        case "c":
          ins='String.fromCharCode('+ins+')';
          fn='String.prototype.padding';
          break;
          
        case "d":
        case "u":
          ins='Number('+ins+')';
          fn='Number.prototype.dec';
          break;
          
        case "f":
          ins='Number('+ins+')';
          fn='String.prototype.padding';
          if ( x[5] )
            ins+='.toFixed('+x[5].substr(1)+')';
          else if ( x[4] )
            ins+='.toExponential('+x[4]+')';
          else
            ins+='.toExponential()';
          
          x[2]=x[2]=="-"? "+": "-"; // Invert sign because this is not number but string
          break;
          
        case "o":
          ins='Number('+ins+')';
          fn='Number.prototype.oct';
          break;
          
        case "s":
          ins='String('+ins+')';
          fn='String.prototype.padding';
          break;
          
        case "x":
          ins='Number('+ins+')';
          fn='Number.prototype.hexl';
          break;
          
        case "X":
          ins='Number('+ins+')';
          fn='Number.prototype.hex';
          break;
        }
      
      return '",'+fn+'.call('+ins+',"'+(x[2]+x[4])+'","'+(x[3].substr(x[3].length-1)||" ")+'"),"';
      });
    
    result='(function(){return function(){return ["'+result+'"].join("");};})()';
    result=result.split(/\r/).join('\\r').split(/\n/).join('\\n');
    
    /* jshint evil:true */
    return eval(result);
    },
  
  /**
   * Function splitLimit(delim,limit)
   *
   * <p>Corrects the result of the standard method like described below:
   * 
   * @example <code>
   * var str = "a b c d e f";
   * var arr = str.split(" ", 3);
   *
   * // standard method
   * // required ["a", "b", "c d e f"]
   * // received ["a", "b", "c"]
   *
   * // modified method
   * // required ["a", "b", "c d e f"]
   * </code>
   * 
   * @type Function
   * 
   * @param {String} delim
   * @param {Number} limit
   * 
   * @return {any}
   * 
   * @see http://forum.dklab.ru/viewtopic.php?p=74826
   *      http://msdn.microsoft.com/library/default.asp?url=/library/en-us/jscript7/html/jsmthsplit.asp
   *      http://wdh.suncloud.ru/js09.htm#hsplit
   */
  splitLimit: function(delim,limit)
    {
    if ( !limit && Number(limit)<=0 )
      return this.split(delim,limit);
    
    var isRegExp=delim && delim.constructor==RegExp,res,ref;
    if ( isRegExp )
      {
      res=delim.source;
      ref="";
      if ( delim.ignoreCase )
        ref+="i";
      
      if ( delim.multiline )
        ref+="m";
      }
    else
      {
      res=delim;
      ref="";
      }
    
    var x=this.match(new RegExp("^((?:.*?"+res+"){"+(limit-1)+"})(.*)",ref));
    if ( x )
      {
      var result=x[1].__split__(delim,limit),n=result.length;
      if ( !isRegExp && n )
        --n;
      
      result[n]=x[2];
      return result;
      }
    
    return this.valueOf();
    },
  
  /**
   * Function parseUrl()
   * 
   * <p>Considers the string object as URL and returns it's parts separately.
   * 
   * @type Function
   * 
   * @return {Object} in form of with members as described below.
   * 
   * <p>An input URL as "http://www.ics.uci.edu/pub/ietf/uri/#Related" would give:
   *  <ul>
   *    <li><code>scheme   : </code>"http" (protocol scheme, in lower case)
   *    <li><code>subscheme: </code><i>empty-string</i> in this case
   *    <li><code>host     : </code>"www.ics.uci.edu" (host name) 
   *    <li><code>user     : </code>user, empty string for none 
   *    <li><code>pass     : </code>password, empty string for none 
   *    <li><code>path     : </code>"/pub/ietf/uri/" 
   *    <li><code>port     : </code>{Number} Port number, 80 in this case. Schema "http" gives port 80 and "https" 443 if not specified.  
   *    <li><code>query    : </code><i>empty-string</i> (no query '?' in this case)
   *    <li><code>fragment : </code>"Related" (fragment after '#')
   *  </ul>
   */
  parseUrl: function()
    {
    var matches=this.match(PRE) || {},
        scheme=(matches[1] || '').toLowerCase(),
        port=(scheme=='http' )? 80 :
             (scheme=='https')? 443:
             -1,
        portString=matches[5];
    
    // Parse port, if any.
    if ( portString )
      {
      try { port=Number.parseInt(portString); }
      catch(e) {}
      }
    
    return {
           scheme   : scheme,
           subscheme: matches[2] || '',
           user     : matches[3] || '',
           pass     : matches[4] || '',
           host     : matches[5] || '',
           port     : port,
           path     : matches[7] || '',
           query    : matches[8] || '',
           fragment : matches[9] || ''
           };
    }
  });
  
// Add replaceAll if not present (IE + Samsung Mobile browser).
if ( !String.prototype.replaceAll )
  String.prototype.replaceAll=function(o,n)
    {
    // If plain string, escape special characters to treat it as a literal string.
    if ( typeof o==='string' )
      {
      for ( var s=this,i=s.indexOf(o); i>=0; )
        {
        s=s.substr(0,i)+n+s.substr(i+o.length);
        i=s.indexOf(i+n.length,o);
        }
        
      return s;
      }
       
    // If regular expression, use it directly.
    if ( o instanceof RegExp )
      return this.replace(o,n);

    // Invalid input for oldSubstring
    throw new Error('Invalid first parameter: it must be a string or RegExp');
    };
  
// End of wrap in function.
})();
