/**
 * (C) Copyright Mindus SARL, 2025.
 * All rights reserved.
 *
 * @fileoverview IIZI Service Worker, handles e.g. push notification and
 *               Progressive Web App functionality.
 *
 * @author Christopher Mindus
 */

/* jshint esversion:6, unused:true, undef:true */
/* globals self, console, registration, setTimeout, clearTimeout, clients, firebase */

/**
 * Version of this Service Worker.
 */
const VER='0.2.6',

/**
 * The user agent of the browser.
 */
AGENT=navigator.userAgent,

/**
 * Safari does not support push notifications.
 */
SAFARI=(AGENT.indexOf(' AppleWebKit/')>0),

/**
 * Release version or in-house debugging.
 */
DEBUG=true,

/**
 * Various strings.
 */
PUSH='push';

/**
 * For any case that the notification does not have a timestamp, one will be added
 * by the Service Worker when the notification is received. The options member 'received'
 * is also set to the local time in milliseconds since the Epoch.
 *
 *
 * Possible options:
 * =================
 *        
 * actions:
 * --------
 *   An array of actions object literals to display in the notification.
 *   The members of the array should be an object literal, that itself may contain the
 *   following values:
 *
 *    - action  A DOMString identifying a user action to be displayed on the notification.
 *    - title   A DOMString containing action text to be shown to the user.
 *    - icon    A USVString containing the URL of an icon to display with the action.
 *
 *   Appropriate responses are built using "event.action" within the "notificationclick" event
 *   when the user responds to the notification.
 *
 *   In IIZI, the 'action' string can be:
 *
 *     + an URL starting with 'http://' or 'https://' that will open that page.
 *
 *     + The 'iza:' prefix of the action will direct the request to the client(s) app that will
 *       process the request in the server as fit. If there no client is found, the page or PWA
 *       will be opened, and when started, the request will be placed there to be processed.
 *       Several such requests could potentially be added before the server has time to process
 *       them, thus the server will receive multiple requests. This request is only valid for
 *       normal IIZI PWA's, the iiziRun Developer will only process these if an app is started.
 *       iiziRun Custom will post this to its default app even if it's not running, but it
 *       requires it to be configured for it. For both iiziRun Developer and Custom, this action
 *       will open the app in all cases. 
 *
 *     + The string 'iz:open' will open the first client found, and if none is found, the page
 *       will be opened thus starting the PWA.
 *
 * badge:
 * ------
 *   A USVString containing the URL of an image to represent the notification when there is
 *   not enough space to display the notification itself such as for example, the Android
 *   Notification Bar. On Android devices, the badge should accommodate devices up to
 *   4x resolution, about 96 by 96 px, and the image will be automatically masked.
 *
 * body:
 * -----
 *   A string representing an extra content to display within the notification.
 *   HTML is NOT supported: it must be a plain string.
 *
 * data:
 * -----
 *   Arbitrary data that you want to be associated with the notification.
 *   This can be of any data type. IIZI will process this as being an Object and if any
 *   client is open (and no special 'iza:' or 'iz:' action is chosen), will check the
 *   following:
 *
 *     + inapp = true or false   (Boolean 'truish' or 'falsy')
 *       will show the notification in the app also (depending on the 'show' option below).
 *
 *     + show  = 'server', 'app' or 'none'  (String)
 *       will cause the notification not to be shown using the client's Service Worker
 *       'showNotification' API, relying solely on being processed by the iiziRun's or
 *       the app depending on 'actions' provided. The 'app' will cause the client to
 *       process it, whereas 'server' will do nothing but to pass it on to the server
 *       app for processing as it sees fit. For 'server', the entire notification event
 *       is passed on.
 *
 *     + topic   (String)
 *       will be added to the notification
 *
 * dir:
 * ----
 *   The direction of the notification; it can be auto, ltr or rtl.
 *
 * icon:
 * -----
 *   A USVString containing the URL of an image to be used as an icon by the notification.
 *
 * image:
 * ------
 *   A USVString containing the URL of an image to be displayed in the notification.
 *
 * lang:
 * -----
 *   Specify the lang used within the notification. This string must be a valid BCP 47 language tag.
 *
 * renotify:
 * ---------
 *   A boolean that indicates whether to suppress vibrations and audible alerts when reusing a tag
 *   value. If options’s renotify is true and options’s tag is the empty string a TypeError will be
 *   thrown. The default is false.
 *
 * requireInteraction:
 * ------------------- 
 *   Indicates that on devices with sufficiently large screens, a notification should remain active
 *   until the user clicks or dismisses it. If this value is absent or false, the desktop version of
 *   Chrome will auto-minimize notifications after approximately twenty seconds.
 *   The default value is false.
 *
 * silent:
 * -------
 *   When set indicates that no sounds or vibrations should be made. If options’s silent is true and
 *   options’s vibrate is present a TypeError exception will be thrown. The default value is false.
 *
 * tag:
 * ----
 *   An ID for a given notification that allows you to find, replace, or remove the notification
 *   using a script if necessary.
 *
 * timestamp:
 * ----------
 *   A DOMTimeStamp representing the time when the notification was created. It can be used to
 *   indicate the time at which a notification is actual. For example, this could be in the past
 *   when a notification is used for a message that couldn’t immediately be delivered because the
 *   device was offline, or in the future for a meeting that is about to start.
 *
 * vibrate:
 * --------
 *   A vibration pattern to run with the display of the notification. A vibration pattern can be
 *   an array with as few as one member. The values are times in milliseconds where the even indices
 *   (0, 2, 4, etc.) indicate how long to vibrate and the odd indices indicate how long to pause.
 *   For example, [300, 100, 400] would vibrate 300ms, pause 100ms, then vibrate 400ms.
 */

// Log start of Service Worker.
log('Started version '+VER+' - '+AGENT);

///
/// --- Variables ---
///

/**
 * The Firebase instance, 'undefined' for Safari.
 */
var _firebase,

/**
 * The Firebase messaging instance, initialized for the app.
 */
_messaging,

/**
 * "Boolish" flag indicating the push data has changed and needs writing to cache.
 */
hasChanged=0,

/**
 * Timeout ID used when the settings has changed and needs to be written
 * to disk. The timeout is 5 seconds. The timeoutID is 'falsy' if none
 * is currently running. 
 */
timeoutID,

/**
 * Persistent cache instance.
 */
persistentCache,

/**
 * Volatile cache of most files that always can be reloaded,
 * assuming there is a network connection.
 */
volatileCache,

/**
 * The communication port for message posting.
 */
communicationPort,

/**
 * The current appServer URL.
 *
 * @type String.
 */
appServer,
 
/**
 * The application server key(s). If just one, it's the iiziRun or the app.
 * If there are two, an app has started from iiziRun. If there are more than
 * two, iiziRun has restarted and opened another app.
 *
 * The appServer object has the following members:
 *  + url : The iiziServer URL.
 *  + time: Timestamp when received.
 *  + keys: The keys object, all strings:
 *     'v' for VAPID,
 *     'a' for Apple,
 *     'f' for Firebase.
 */
appServers=[],

/**
 * The UUID for each user and server. Each entry contains an Object with the
 * following members:
 *  + url : The iiziServer URL.
 *  + time: Timestamp when received.
 *  + user: The user ID.
 *  + epw : The encoded password (that was in either in clear text or "hashed"
 *          password from server).
 *
 * @type Array
 */
userUUIDs=[];


///
/// --- Local functions ---
///

/**
 * Function log(msg,...args)
 *
 * @type Function
 *
 * @param {String} msg  Message string to log.
 */
function log()
  {
  var args=Array.from(arguments);
  args[0]='iiziSW-'+VER+': '+args[0];
  console.log.apply(console,args);  
  }

/**
 * Function initFirebase(config)
 *
 * @type Function
 *
 * @param {Object} config  The Firebase config.
 */
function initializeFirebase(config)
  {
  // Import 'Firebase' app and messaging required for push notification.
  if ( !SAFARI )
    {
    // Import Firebase Messaging if it's not Safari.
    importScripts('https://www.gstatic.com/firebasejs/8.3.0/firebase-app.js',
                  'https://www.gstatic.com/firebasejs/8.3.0/firebase-messaging.js');

    // Save the _firebase instance.
    _firebase=firebase;
    }
  }
      
/**
 * Function setChanged()
 *
 * <p>Marks the settings as changed and will invoke the flush function in 5 seconds using
 * a timeout.
 */
function setChanged()
  {
  hasChanged=1;
  if ( timeoutID )
    clearTimeout(timeoutID);

  timeoutID=setTimeout(flush,5000);
  }
  
/**
 * Function reset()
 *
 * <p>Resets the Service Worker to its initial state, requested from iiziRun or the server.
 *
 * @type Function
 */
function reset()
  {
  log('reset');
  appServers=[];
  userUUIDs=[];
  
  
  hasChanged=true;
  flush();
  }
  
/**
 * Function flush()
 *
 * <p>Write the settings to the cache now.
 *
 * @type Function
 */
function flush()
  {
  log('flush: changed = '+(!!hasChanged));
  if ( hasChanged )
    {
    // Build new settings.
    var settings=
      {
      servers: appServers,
      users  : userUUIDs
      };

    // Put new settings in the cache.
 
    // Reset changes and clear potential timeout.
    if ( timeoutID )
      clearTimeout(timeoutID);
      
    hasChanged=timeoutID=0;
    }
  }

/**
 * Function cleanCache(keepUUIDs)
 *
 * <p>Cleans up the cache.
 *
 * @type Function
 *
 * @param {Boolean} keepUUIDs  Keep the existing UUIDs if true, false clears that too.
 */
function cleanCache(keepUUIDs)
  {
  log('clean cache, keep UUIDs = '+keepUUIDs);
  flush();
  }
 
/**
 * Function isObject(obj)
 *
 * <p>Checks if the parameter is an Object.
 *
 * @param obj  Any type of parameter.
 *
 * @return {Boolean} true if it's an {Object} that is non-null, false otherwise.
 */
function isObject(obj)
  {
  var type=typeof obj;
  return type==='function' || type==='object' && !!obj;
  }

/**
 * Function deepEquals(o1,o2)
 *
 * <p>Deep-equals checks if two objects are equal with it's members.
 * Support for members like Boolean, String and Number: no others are required here.
 *
 * @type Function
 *
 * @return {Boolean} truish if they are equal, falsy otherwise. 
 */
function deepEquals(o1,o2)
  {
  if ( isObject(o1)===isObject(o2) )
    {
    var checked={},p;

    // First Object members.
    for ( p in o1 )
      if ( deepEquals(o1[p],o2[p]) )
        checked[p]=true;
      else
        return false;
        
    // Check second Object members, except the ones already checked.
    for ( p in o2 )
      if ( !checked[p] && !deepEquals(o1[p],o2[p]) )
        return false;
 
    return true;
    }

  return (o1!==o2);
  }

/**
 * Function onServerKeys(data)
 *
 * <p>Processes the event when a new or updated server keys has been received
 * by a client. This will modify the 'appServers' array by changing an entry
 * or adding a new one. Modifications will be set to changed, thus flushed to
 * cache a little later.
 *
 * <p>The 'data' has the following members saved in the very short 'appServers'
 * {Object}:
 * 
 * <ul>
 *   <li>'u' = url - the server URL,
 *   <li>'a' = appID - the App ID ('*' for global or iiziRun),
 *   <li>'k' = 'keysObject' as described below,
 *   <li>'t' = time - timestamp when created or updated.
 * </ul>
 *
 * <p>The 'keysObject' is an {Object} with the following potential members:
 * <ul>
 *   <li>'v'  = VAPID-based provider using Service Worker,
 *   <li>'fc' = Firebase Cloud Messaging in Cordova,
 *   <li>'fw' = Firebase Cloud Messaging of the Web,
 *   <li>'i'  = iOS in Cordova, and
 *   <li>'a'  = Apple Push in Safari (could be macOS and iOS without Cordova).
 * </ul>
 *
 * @type Function
 *
 * @param {Object}  Server keys object.
 */
function onServerKeys(data)
  {
  console.warn('onServerKeys = '+JSON.stringify(data,null,2));
  for ( var url=data.url,
            keys=data.keys,
            ii=appServers.length,
            timestamp=Date.now(),
            arrayKeys;
        ii--; )
    if ( (arrayKeys=appServers[ii]).url==url )
      {
      if ( !deepEquals(keys,arrayKeys) )
        {
        log('update server keys = '+JSON.stringify(
          appServers[ii]=
            {
              
              // TODO: MERGE KEYS ???
              
            u: url,
            k: keys,
            t: timestamp
            },null,2));

        setChanged();
        }

      return;
      }
      
  appServers.push(ii=
    {
    u: url,
    k: keys,
    t: timestamp
    });
 
  setChanged();
  log('add server keys = '+JSON.stringify(ii,null,2));
  }
      
/**
 * Function onUserInfo(data)
 *
 * <p>Processes the event when a new or updated user with its datahas been received
 * by a client. This will modify the 'userUUIDs' array by changing an entry
 * or adding a new one. Modifications will be set to changed, thus flushed to
 * cache a little later.
 *
 * <p>The 'data' has the following members saved in the very short ''
 *
 *          // Key values.
          url  : server,
          user : userID,

          // Additional data.
          epw  : epw,
          uuid : _uuid,
          agent: _userAgent
          
 * @type Function
 *
 * @param {Object}  Server keys object.
 */
function onUserInfo(data)
  {
  console.warn('onUserInfo = '+JSON.stringify(data,null,2));
   /*
  for ( var url=data.url,
            user=data.user,
            epw=data.epw,
            agent=data.agent,
            ii=userUUIDs.length,
            value=
              {
              url  : url,
              agent: data.agent,
              user : user,
              epw  : data.epw,
              time : Date.now()
              },
            U; ii--; )
        {
        U=userUUIDs[jj];
        if ( u==U.url && user==U.user )
          {
          // Updated?
          if ( agent!=)
          return;
          }
          }
          
  for ( var url=data.url,
            k=data.keys,
            ii=appServers.length,
            v=
              {
              url: url,
              time: Date.now(),
              keys: keys
              },
            u, p, K;
        ii--; )
    if ( (K=appServers[ii]).url==url )
      {
      if ( !deepEquals(k,K) )
        log('update server keys = '+JSON.stringify(appServers[ii]=v,null,2));

      return;
      }
      
  appServers.push(v);
  log('add server keys = '+JSON.stringify(v,null,2));*/
  }
  
///
/// --- Event listeners ---
///

/**
 * Add custom event listener "iiziAppServer" for special message from iiziRun or
 * the iiziApp to define the app server keys for subscriptions.
 */
self.addEventListener('message',e=>
  {
  var d=e.data;
  if ( d )
    switch(d.type)
      {
      // Communication port initialization.
      case 'port':
        log('communication port initialized, client = '+d.p+' ('+d.n+') version '+d.v+' built '+new Date(d.t));
        (communicationPort=e.ports[0]).postMessage(
          {
          type: 'ver',
          v: VER
          });      
        break;
  
      // User being set with password and device info ("User Agent" and UUID).
      // The object is 'url', 'user', 'epw', 'agent' and 'uuid'.
      // This will also set the current server to the specified URL.
      case 'user':
        onUserInfo(d);
        break;
        
      // Server keys: save them.
      case 'keys':
        onServerKeys(d);
        break;
        
      // Flush the settings data to cache now.
      case 'flush':
        flush();
        break;
        
      // Clean up cache.
      case 'clean':
        cleanCache(true);
        break;
   
      // Reset: this means tutti-frutti!
      case 'reset':
        reset();
        break;
   
      // Others are not handled.
      default:
        console.error('Received unknown message',d);
      }
  });

/**
 * Handle install to transfer vital data in a previous cache to this
 * latest one. It should only be one previous version though, but you never
 * know. So the first one encountered will be the data to transfer. The
 * data is stored in a JSON object file called "instance.json".
 */  
self.addEventListener('install',e=>
  {
  log('install version '+VER);
  
  // Open the persistent cache.
  //e.waitUntil(caches.open('')

  
  e.waitUntil(self.skipWaiting());
  });

/**
 * We handle the activation event by taking control of every client
 * present in the system.
 */
self.addEventListener('activate',e=>
  {
  log('activate version '+VER);
  console.log('Clients = ',clients);
  e.waitUntil(clients.claim());      
  });
  
/**
 * Handles the notification of an updated Service Worker. This means
 * that another newer version would soon be installed, so flush out
 * any pending changes to cache NOW.
 */
self.addEventListener('onupdatefound',e=>
  {
  log('onupdatefound, current version = '+VER+' -> ',e);
  flush();
  });
  
// Different support for Safari push notifications.
if ( SAFARI )
  {
  // TODO!
  log('installing Safari browser support');
  }
else
  {
  // Other browsers support Push API.
  log('installing Push API support');

  /**
   * Handle subscription change, very buggy and most browsers, etc doesn't
   * handle it correctly. So just let us assume the worst.
   */
  self.addEventListener('pushsubscriptionchange',e=>
    {
    var old=e.oldSubscription,options;
    
    if ( old && (options=old.options) )
      e.waitUntil(registration.pushManager.subscribe(options)
        .then(sub=>
          {  
          // TODO: Send new subscription to application server.
          console.warn('Got new subscription = ',sub);  
          })
        .catch(e=>
          {
          console.warn('pushsubscriptionchange error (expected!)',e);
          }));
    });
   
  /**
   * Handles push event notification.
   */
  self.addEventListener(PUSH,e=>
    {
    var d=self.Notification,title,options;
    var paramCount=0;
   
    // Only display notification if permission for push notification is granted.
    if ( d && d.permission=='granted' )
      {
      if ( (d=e.data) )
        {
        // Convert to data object using JSON if possible, otherwise use text.
        try
          {
          options=d.json();
          title=options.title;
          }
        catch(x)
          {
          options={};
          try { title=d.text(); }
          catch(y) {}
          }
  
        console.log('Notification received: '+title,d);
        
        // TODO: Remove before release! 
        // TODO: Remove before release! 
        // TODO: Remove before release! 
        // TODO: Remove before release! 
        // Count the parameters.
        if ( DEBUG )
          for ( d in options )
            ++paramCount;
        // TODO: Remove before release! 
        // TODO: Remove before release! 
        // TODO: Remove before release! 
        // TODO: Remove before release!
        
  title='Les applis streamées arrivent à Monaco';       
   
        // Set or overwrite title in options.
        options.title=title=title || PUSH;
   
        // Add timestamp if not present.
        options.received=d=Date.now();
        if ( !options.timestamp )
          options.timestamp=d;
  
        // Set Booleans correctly.
        ['renotify','requireInteraction','silent'].forEach(p=>
          {
          if ( p in options )
            options[p]=!!options[p];
          });
   
        // Set the options with the icon if not present.
        if ( !options.icon )
          options.icon='images/msg/info.png';
  
        // TODO: Remove before release! 
        // TODO: Remove before release! 
        // TODO: Remove before release! 
        // TODO: Remove before release! 
        // Set options if only title was found.
        if ( DEBUG && !paramCount && !options.body && !options.image )       
          options=
            {
  //          icon : 'images/96.png',
  icon: 'https://support.iizi.co/images/monaco-info-96x96.png',
  body: 'Avec Mindus et IIZI, vos applis sont désormais streamées et sécurisées.',
  //          body : 'Long text that appears in the body or message area, sometimes not fitting inside the notification area, so it is made smaller, generally truncated. However, sometimes it is possible that the device or browser allows to display it in full. [The End]',
  //          image: 'https://secureservercdn.net/160.153.137.210/5a9.c9e.myftpupload.com/wp-content/uploads/2017/12/1.1a.png',
  image: 'images/stream.jpg',
            requireInteraction: true,
            actions:
              [
              { icon: 'images/96.png', title: 'Voir reportage', action: 'idOpen'    },
              { icon: 'https://support.iizi.co/images/monaco-info-96x96.png', title: 'Ouvir Monaco Info', action: 'idMC' } 
  /*          { icon: 'images/msg/warn.png' , title: 'Open iiziRun app', action: 'idOpen'    }, 
              { icon: 'images/msg/error.png', title: '* Delete *'      , action: 'idDelete'  }, 
              { icon: 'images/msg/info.png' , title: 'Dismiss me now'  , action: 'idDismiss' }*/ 
              ]
            };
        // TODO: Remove before release! 
        // TODO: Remove before release! 
        // TODO: Remove before release! 
        // TODO: Remove before release!
        
        // Send to client for processing.
        if ( communicationPort )
        +
          communicationPort.postMessage(
            {
            type: PUSH,
            p: options
            }); 
            
        // Display notification and wait until removed.
        // TODO: handle onClick? 
        e.waitUntil(self.registration.showNotification(title,options));
        }
      }
    });
  }
  
/*

READ MORE AT https://developers.google.com/web/fundamentals/push-notifications/common-notification-patterns


Notification close event
In the last section we saw how we can listen for notificationclick events.

There is also a notificationclose event that is called if the user dismisses one of your notifications (i.e. rather than clicking the notification, the user clicks the cross or swipes the notification away).

This event is normally used for analytics to track user engagement with notifications.


self.addEventListener('notificationclose', function(event) {
  const dismissedNotification = event.notification;

  const promiseChain = notificationCloseAnalytics();
  event.waitUntil(promiseChain);
});


Open a window
One of the most common responses to a notification is to open a window / tab to a specific URL. We can do this with the clients.openWindow() API.

In our notificationclick event, we'd run some code like this:


const examplePage = '/demos/notification-examples/example-page.html';
const promiseChain = clients.openWindow(examplePage);
event.waitUntil(promiseChain);

Focus an existing window
When it's possible, we should focus a window rather than open a new window every time the user clicks a notification.

Before we look at how to achieve this, it's worth highlighting that this is only possible for pages on your origin. This is because we can only see what pages are open that belong to our site. This prevents developers from being able to see all the sites their users are viewing.

Taking the previous example, we'll alter the code to see if /demos/notification-examples/example-page.html is already open.


const urlToOpen = new URL(examplePage, self.location.origin).href;

const promiseChain = clients.matchAll({
  type: 'window',
  includeUncontrolled: true
}).then((windowClients) => {
  let matchingClient = null;

  for (let i = 0; i < windowClients.length; i++) {
    const windowClient = windowClients[i];
    if (windowClient.url === urlToOpen) {
      matchingClient = windowClient;
      break;
    }
  }

  if (matchingClient) {
    return matchingClient.focus();
  } else {
    return clients.openWindow(urlToOpen);
  }
});

event.waitUntil(promiseChain);

 */