        // Service Worker only has a single subscription, but ask the worker as we handle
        // 'UserID', 'AppID' and 'Topic', and this will "emulate" unsubscription of all.
        // Then the Server must be informed that this device is unsubscribing all of the
        // subscriptions (please note that the server might not be initialized), so the
        // Service Worker will return a list of Servers that must be informed, and with
        // what User ID and encrypted password to use.

/**
 * (C) Copyright Mindus SARL, 2025.
 * All rights reserved.
 *
 * @fileoverview Push notification Support.
 *
 * <p>For any case that the notification does not have a timestamp, one will be added
 * by the Service Worker when the notification is received. The options member 'received'
 * is also set to the local time in milliseconds since the Epoch.
 *
 *
 * <h2>Possible options:</h2>
 *        
 * <p><b>actions:</b>
 * <br>An array of actions object literals to display in the notification.
 * The members of the array should be an object literal, that itself may contain the
 * following values:
 * <ul>
 *   <li>action - A DOMString identifying a user action to be displayed on the notification.
 *   <li>title  - A DOMString containing action text to be shown to the user.
 *   <li>icon   - A USVString containing the URL of an icon to display with the action.
 * </ul>
 *
 * <p>Appropriate responses are built using "event.action" within the "notificationclick" event
 * when the user responds to the notification.
 *
 * <p>In IIZI, the 'action' string can be:
 *
 * <ul>
 * <li>an URL starting with 'http://' or 'https://' that will open that page.
 *
 *   <li>The 'iza:' prefix of the action will direct the request to the client(s) app that will
 *       process the request in the server as fit. If there no client is found, the page or PWA
 *       will be opened, and when started, the request will be placed there to be processed.
 *       Several such requests could potentially be added before the server has time to process
 *       them, thus the server will receive multiple requests. This request is only valid for
 *       normal IIZI PWA's, the iiziRun Developer will only process these if an app is started.
 *       iiziRun Custom will post this to its default app even if it's not running, but it
 *       requires it to be configured for it. For both iiziRun Developer and Custom, this action
 *       will open the app in all cases. 
 *
 *   <li>The string 'iz:open' will open the first client found, and if none is found, the page
 *       will be opened thus starting the PWA.
 * </ul>
 *
 * <p><b>badge:</b>
 * <br>A USVString containing the URL of an image to represent the notification when there is
 * not enough space to display the notification itself such as for example, the Android
 * Notification Bar. On Android devices, the badge should accommodate devices up to
 * 4x resolution, about 96 by 96 px, and the image will be automatically masked.
 *
 * <p><b>body:</b>
 * <br>A string representing an extra content to display within the notification.
 * HTML is NOT supported: it must be a plain string.
 *
 * <p><b>data:</b>
 * <br>Arbitrary data that you want to be associated with the notification.
 * This can be of any data type. IIZI will process this as being an Object and if any
 * client is open (and no special 'iza:' or 'iz:' action is chosen), will check the
 * following:
 * <ul>
 *   <li><i>inapp</i> = true or false   (Boolean 'truish' or 'falsy')
 *       <br>will show the notification in the app also (depending on the 'show' option below).
 *
 *   <li><i>show</i>  = 'server', 'app' or 'none'  (String)
 *       <br>will cause the notification not to be shown using the client's Service Worker
 *       'showNotification' API, relying solely on being processed by the iiziRun's or
 *       the app depending on 'actions' provided. The 'app' will cause the client to
 *       process it, whereas 'server' will do nothing but to pass it on to the server
 *       app for processing as it sees fit. For 'server', the entire notification event
 *       is passed on.
 *
 *   <li><i>topic</i>   (String)
 *       <br>will be added to the notification
 * </ul>
 *
 * <p><b>dir:</b>
 * <br>The direction of the notification; it can be auto, ltr or rtl.
 *
 * <p><b>icon:</b>
 * <br>A USVString containing the URL of an image to be used as an icon by the notification.
 *
 * <p><b>image:</b>
 * <br>A USVString containing the URL of an image to be displayed in the notification.
 *
 * <p><b>lang:</b>
 * <br>Specify the lang used within the notification. This string must be a valid BCP 47 language tag.
 *
 * <p><b>renotify:</b>
 * <br>A boolean that indicates whether to suppress vibrations and audible alerts when reusing a tag
 * value. If options’s renotify is true and options’s tag is the empty string a TypeError will be
 * thrown. The default is false.
 *
 * <p><b>requireInteraction:</b>
 * <br>Indicates that on devices with sufficiently large screens, a notification should remain active
 * until the user clicks or dismisses it. If this value is absent or false, the desktop version of
 * Chrome will auto-minimize notifications after approximately twenty seconds.
 * The default value is false.
 *
 * <p><b>silent:</b>
 * <br>When set indicates that no sounds or vibrations should be made. If options’s silent is true and
 * options’s vibrate is present a TypeError exception will be thrown. The default value is false.
 *
 * <p><b>tag:</b>
 * <br>An ID for a given notification that allows you to find, replace, or remove the notification
 * using a script if necessary.
 *
 * <p><b>timestamp:</b>
 * <br>A DOMTimeStamp representing the time when the notification was created. It can be used to
 * indicate the time at which a notification is actual. For example, this could be in the past
 * when a notification is used for a message that couldn’t immediately be delivered because the
 * device was offline, or in the future for a meeting that is about to start.
 *
 * <p><b>vibrate:</b>
 * <br>A vibration pattern to run with the display of the notification. A vibration pattern can be
 * an array with as few as one member. The values are times in milliseconds where the even indices
 * (0, 2, 4, etc.) indicate how long to vibrate and the odd indices indicate how long to pause.
 * For example, [300, 100, 400] would vibrate 300ms, pause 100ms, then vibrate 400ms.
 *
 *
 * <h2>Return codes for the Push API</h>
 *
 * <p>There are similarities with the Push Notification State, where 1 = Granted, 0 = Unconfigured
 * or Default, -1 = Denied and -2 = No support for push notifications on the device/browser.
 *
 * <p>These return codes are used when the client communicates with the server, and sometimes,
 * depending on the Java API, these return codes will be used.
 *
 * <ul>
 *   <li>   0 = SUCCESS
 *   <li>  -1 = DENIED
 *   <li>  -2 = UNSUPPORTED
 *
 *   <li>  10 = POST_FAILED
 *              <br>Can be accompanied with an additional HTTP response code, or zero if none applies.
 *   <li>  11 = NO_POST_DATA_REPLY
 *   <li>  12 = INVALID_JSON_REPLY
 *              <br>The server replied with invalid JSON to the PUT or GET request.
 *
 *   <li>  20 = NO_SERVER
 *   <li>  21 = INVALID_SERVER
 *   <li>  22 = NO_USER
 *   <li>  23 = INVALID_APPID
 *   <li>  24 = INVALID_TOPIC
 *   <li>  25 = NO_SERVICE_WORKER_CONTROLLER
 *
 *   <li>  30 = GET_FIREBASE_PERMISSION_FAILED
 *   <li>  31 = GET_FIREBASE_ID_FAILED
 *   <li>  32 = GRANT_FIREBASE_PERMISSION_FAILED
 *   <li>  33 = GET_FIREBASE_ANDROID_TOKEN_FAILED
 *   <li>  34 = GET_FIREBASE_APNS_TOKEN_FAILED
 *   <li>  35 = GET_FIREBASE_WEB_TOKEN_FAILED
 *   <li>  36 = GET_SAFARI_TOKEN_FAILED
 *   <li>  37 = REQUEST_VAPID_PERMISSION_FAILED
 *   <li>  38 = GET_SUBSCRIPTIONS_FAILED
 *   <li>  39 = GET_VAPID_SUBSCRIPTION_FAILED
 *
 *   <li> 100 = NO_VAPID_KEY
 *   <li> 101 = NO_FIREBASE_ANDROID_KEY
 *   <li> 102 = NO_FIREBASE_APNS_KEY
 *   <li>
 *
 *   <li> 120 = NO_FIREBASE_ANDROID_TOKEN
 *   <li> 121 = NO_FIREBASE_APNS_TOKEN
 *   <li> 122 = NO_FIREBASE_WEB_TOKEN  
 *
 *   <li> 130 = GET_VAPID_SUBSCRIPTION_FAILED
 * </ul>
 *
 *
 * <h2>Testing Send Push Notifications</h2>
 *
 * <ul>
 *   <li><b>Firebase:/b>
 *       <br>In a browser, login to the Firebase Console and select you app. Then append
 *       "/notifications" to the browser URL. Then follow instructions on the web.
 * </ul>
 *
 *
 * @author Christopher Mindus
 */

/* jshint esversion:6, unused:true, undef:true */
/* globals window, navigator, location, console, XMLHttpRequest, MessageChannel, btoa,
   addEventListener, decodeURIComponent, escape, log, _mixin, device, LogError, LogWarn,
   UUID 
*/

(function(window,navigator)
  {
  // Defines that JavaScript code should be executed in "strict mode".
  "use strict";
  
  ///
  /// --- Constants ---
  ///

  // Debug flag.  
  const DEBUG=1,
  
  // Arguments to array call.
  toArray=[].slice, 
 
  // Saved origin of the 'index.html' (or "iiziApp.html') that started the app.
//  _origin=location.origin,
 
  // Permission strings.
  S_DEFAULT     = 'default',
  S_DENIED      = 'denied',
  S_GRANTED     = 'granted',

  // Permission values.  
  GRANTED      =  1,
  UNCONFIGURED =  0,
  DENIED       = -1,
  UNSUPPORTED  = -2,

  ///
  /// --- Error codes ---
  ///
  
  SUCCESS                            =   0,
  
  POST_FAILED                        =  10,
  NO_POST_DATA_REPLY                 =  11,
  INVALID_JSON_REPLY                 =  12,
  
  NO_SERVER                          =  20,
  INVALID_SERVER                     =  21,
  NO_USER                            =  22,
  INVALID_APPID                      =  23,
  INVALID_TOPIC                      =  24,
  NO_SERVICE_WORKER_CONTROLLER       =  25,
//  GET_SERVER_KEYS_FAILED             =  26,
  
//  GET_FIREBASE_PERMISSION_FAILED     =  30,
  GET_FIREBASE_ID_FAILED             =  31,
  GRANT_FIREBASE_PERMISSION_FAILED   =  32,
//  GET_FIREBASE_ANDROID_TOKEN_FAILED  =  33,
//  GET_FIREBASE_APNS_TOKEN_FAILED     =  34,
//  GET_FIREBASE_WEB_TOKEN_FAILED      =  35,
  REQUEST_FIREBASE_PERMISSION_FAILED =  36,    
  REQUEST_VAPID_PERMISSION_FAILED    =  37,
//  GET_SUBSCRIPTIONS_FAILED           =  38,
  GET_VAPID_SUBSCRIPTION_FAILED      =  39,
  
//  NO_VAPID_KEY                      = 100,
//  NO_FIREBASE_ANDROID_KEY           = 101,
//  NO_FIREBASE_APNS_KEY              = 102,

//  NO_FIREBASE_ANDROID_TOKEN         = 120,
//  NO_FIREBASE_APNS_TOKEN            = 121,
//  NO_FIREBASE_WEB_TOKEN             = 122,
  
//  GET_VAPID_SUBSCRIPTION_FAILED     = 130,

  ///
  /// --- Constants (like variables, but they can be initialized once) ---
  ///
  
  // {ServiceWorker} The Service Worker.
  _serviceWorker=navigator.serviceWorker,
    
  // The iiziRun instance.
  _izRun=window.izRun,
  
  // Flag for Cordova presence.
  hasCordova=window.cordova;
    
  ///
  /// --- Variables ---
  ///

  // The 'user agent' string for Web Browsers, for Cordova it's made
  // up from the device information.
  var _userAgent=navigator.userAgent,
  
  // {Notification} The notification instance.
  _notification,
 
  // Firebase in Cordova.
  _firebaseCordova,
  
  // Firebase for Web.
  _firebaseWeb,
 
  // Safari Browser under macOS or iOS when not using Cordova.
  _safariWeb,
  
  // Safari Browser push notification instance under macOS or iOS when not using Cordova.
  _safariPush,
  
  // The {MessageChannel}.
  _messageChannel,
  
  // Callback function when Service Worker posts a reply on the MessageChannel port2.
  _cbPostTransaction,
 
  // Cordova iOS flag.
  _iOS,
    
  // User ID.
  _userID,
    
  // User's password in clear text or a hashed version.
  _password,
  
  // The UUID of this instance, undefined when neither
  // Cordova or Service worker is present.
  _uuid='',
  
  // The server keys Object of the current server for each 'appID' or '*' for all.
  // When the server changes, this value is reset.
  _serverKeys,

  // The Cordova App ID allocated by Firebase.
  _cordovaFirebaseID,
  
  // The Cordova Firebase Token.
  _cordovaFirebaseToken,
  
  // The iOS Cordova APNS token.
  _cordovaTokenAPNS,
  
  // {ServiceWorkerRegistration} The Service Worker registration.
  _swPush,
  
  // Firebase messaging global instance.
  _firebaseWebMessagingGlobal,

  // The Firebase web token.
  _firebaseWebToken,
  
  // Safari Website Push ID, only set when a server is selected.
  _safariWebsitePushID,
  
  // The Safari Web token.
  //_safariToken,
  
  // The VAPID {Object} subscriptions objects per App ID. This is an {Object} with
  // members of the 'appID' or '*' used, where the value is a {PushSubscription}.
  _vapidSubscriptions,
  
  // The latest Firebase cordova subscription.
  _firebaseCordovaSubscription,
  
  // The latest Firebase cordova subscription.
  //_firebaseWebSubscription,
  
  // The latest Safari Web subscription.
  _safariSubscription,
  
  // The server to use.
  server,
  
  // Flag for localhost.
  isLocalhost,
  
  // Flag for secure connection to server.
  isSecure;
  
  ///
  
  /**
   * Function logInfo(...args)
   *
   * <p>Logs an informational message, any number of arguments are allowed, of any type.
   * The log entry is prefixed with 'PushNotification'.
   *
   * @type Function
   */
  function logInfo()
    {
    var a=toArray.call(arguments);
    a.unshift('PushNotification');
    log.apply(window,a);
    }
 
  /**
   * Function logErr(...args)
   *
   * <p>Logs an error, any number of arguments are allowed, of any type.
   * The log entry is prefixed with 'PushNotification error'.
   *
   * @type Function
   */
  function logWarn()
    {
    // If LogWarn is assigned, use that one: it will send to Server.
    var a=toArray.call(arguments);
    a.unshift('PushNotification warning');
    LogWarn.apply(window,a);
    }

  /**
   * Function logErr(...args)
   *
   * <p>Logs an error, any number of arguments are allowed, of any type.
   * The log entry is prefixed with 'PushNotification error'.
   *
   * @type Function
   */
  function logErr()
    {
    // If LogError is assigned, use that one: it will send to Server.
    var a=toArray.call(arguments);
    a.unshift('PushNotification error');
    LogError.apply(window,a);
    }

  /*
   * Function getFirebaseWebMessaging(keys,appID)
   *
   * <p>Returns the Firebare Messaging instance for the App ID.
   *
   * @type Function
   *
   * @param {Object} keys   The server keys for the App ID.
   * @param {String} appID  The App ID.
   *
   * @return An existing Firebase App Messaging instance, a new one
   *         (cached) or 'falsy' for none.
   *
  function getFirebaseWebMessaging(keys,appID)
    {
    var m=_firebaseWebMessagingGlobal;
    if ( server!=_origin || appID!='*' )
      {
      // Are "keys" (i.e. config object) and Firebase Web present?
      if ( keys.w && _firebaseWeb )
        {
        // Get a cached instance or create a new one.
        m=_izRun.initFBWMsg(server+','+appID,keys.w);
        }
      else
        m=0;
      }
      
    return m; 
    }*/

  /**
   * Function getState(cbReply,cbError)
   *
   * <p>Returns the permissions state string in the callback.
   *
   * @type Function
   *
   * @param {Function} cbReply  Callback with a function that will receive a {String}
   *                            as 'granted', 'denied', 'default' or falsy when not unconfigured.
   * @param {Function} cbError  Callback with a {Number} return code and a error message
   *                            {String} in case this fails.
   */
  function getState(cbReply,cbError)
    {
    // Error function.
    function error(code,msg)
      {
      logErr(msg,code);
      cbError(code,msg);
      }
 
    // Unsupported.
    function unsupported()
      {
      error(UNSUPPORTED,'Notifications are not supported.');
      }
      
    // Firebase with Cordova. 
    function doFirebaseCordova()
      {
      _firebaseCordova.hasPermission(
        function(ok)
          {
          if ( ok )
            {
            if ( DEBUG )
              logInfo('Cordova Firebase permission "granted", existing token: ',
                      (_cordovaFirebaseToken || _cordovaTokenAPNS)); 

            cbReply(S_GRANTED);
            }
          else
            {
            if ( DEBUG )
              logInfo('Cordova Firebase permission "denied", existing token: ',
                      (_cordovaFirebaseToken || _cordovaTokenAPNS)); 

            cbReply(S_DENIED);
            }
          },
        function(err)
          {
          error(REQUEST_FIREBASE_PERMISSION_FAILED,'Failed getting Firebase permission state: '+err);  
          });
      }
      
    // Safari Web Browser (iOS and macOS) without Cordova.
    function doSafariWeb()
      {
      if ( _safariPush )
        {
        if ( _safariWebsitePushID )
          {
          var s=_safariPush.permission(_safariWebsitePushID).permission;
          if ( DEBUG )
            logInfo('Safari Web permission: ',_safariWebsitePushID,s);

          cbReply(s);
          }
        else
          {
          if ( DEBUG )
            logWarn('Server must be specified for Safari Web Push Notification');
 
          cbReply(S_DEFAULT);
          }
        }
      else
        unsupported();
      }
      
    // Safari Web or Service Worker use _notification.permission.
    function doServiceWorker()
      {
      if ( DEBUG ) 
        logInfo('Service Worker notification permission: ',_notification.permission);

      cbReply(_notification.permission);
      }

    // Get the server keys.
    getServerKeys('*')
      .then(function(/*keys*/)
        {
        if ( _firebaseCordova )
          doFirebaseCordova();
        else
        if ( _safariWeb )
          doSafariWeb();
        else
        if ( _notification && (_firebaseWeb || _swPush) )
          doServiceWorker();
        else
          unsupported();
        })
      .catch(function(e)
        {
        error(e.code,'Get permission state failed: '+e.message);
        });
    }
  
  /**
   * Function perm(p)
   *
   * <p>Returns the state for a permission string.
   *
   * @type Function
   *
   * @param {DOMString} p String with the permission.
   *
   * @return {Number} 1=Granted, 0=Unconfigured or -1=Denied.
   */
  function perm(p)
    {
    return (p==S_GRANTED)? GRANTED:
           (p==S_DENIED )? DENIED : UNCONFIGURED;
    }
    
  /**
   * Function onPush(push,cbNotify)
   *
   * <p>Notification on this client when a push notification
   * is received.
   *
   * @type Function
   *
   * @param {Object}   push      The push message.
   * @param {Function} cbNotify  Callback notifier function.   
   */
  function onPush(push,cbNotify)
    {
    if ( DEBUG )
      logInfo('onPush: '+JSON.stringify(push,null,2));
 
    if ( cbNotify )
      cbNotify(push);
    }
 
  /**
   * Function callReject(reject,code,msg,obj)
   *
   * <p>Calls the 'reject' function with an {Error} that has the message
   * 'msg' and a member 'code' with the error code. The optional 'obj'
   * members will be added to the error if defined. Be careful NOT to use
   * any defined members of {Error}. 
   *
   * @type Function
   *
   * @param {Function} reject  The Promise reject function that will be
   *                           called with an {Error} object with the
   *                           {Number} member 'code' set to the error code.
   * @param {Number}   code    The error code.
   * @param {String}   msg     The message}.
   * @param {Object}   obj     Optional object to mixin to the Error.
   */
  function callReject(reject,code,msg,obj)
    {
    var e=new Error(msg);
    e.code=code;
    if ( obj )
      _mixin(e,obj);
 
    reject(e);
    }

  /**
   * Function post(func,obj,auth,cb)
   *
   * <p>Posts a Push operation 'code' with the JSON 'obj' to the server
   * at 'host' on 'port' using 'ssl'. The callback 'cb' is invoked with
   * the two parameters: 'error String' (undefined for OK), and JSON {Object}
   * (undefined for error).
   *
   * @type Function
   *
   * @param {String}   func  The push function to perform, appended to the URL
   *                         as "url+'/iizi-push/'+func".
   * @param {Object}   obj   The data Object to post as JSON.
   * @param {Boolean}  auth  Truish if Basic authentication credentials
   *                         should be sent or not (only over secure
   *                         connection), 'falsy' otherwise.
   * @param {Function} cb    Callback function for reply or error:
   *
   * <ol>
   *   <li>The first parameter holds the return code {NUMBER}.
   *   <li>The second parameter is the HTTP response code, set to zero if none is present.
   *   <li>The third parameter is the error message {String} that is 'falsy' when the reply is OK.
   *   <li>The fourth parameter is the reply JSON data that is undefined in case return code is not SUCCESS.
   * </ol>
   */
  function post(func,obj,auth,cb)
    {
    var url=server+'/iizi-push/'+func,
        xhr=new XMLHttpRequest(),
        rc,s;

    function onError()
      {
      if ( xhr )
        {
        if ( (rc=xhr.status || 0) )
          s='Server '+url+' operation failed: '+xhr.status+' '+xhr.statusText;
        else
          s='Invalid Server '+url+' response';

        logErr(s);
        cb(POST_FAILED,rc,s);
        xhr=0;
        }
      }
 
    xhr.onerror=onError;
    xhr.onload=function()
      {
      if ( xhr && xhr.readyState==4 )
        {
        if ( xhr.status==200 )
          {
          s=xhr.responseText;
          if ( DEBUG )
            logInfo('POST reply ('+url+') = '+s);

          if ( s )
            {
            try
              {
              rc=JSON.parse(xhr.responseText);
              }
            catch(e)
              {
              logErr(s='Invalid Server '+url+' response JSON',e);
              cb(INVALID_JSON_REPLY,0,s);
              return;
              }
   
            cb(SUCCESS,0,200,rc);
            }
          else
            {
            logErr(s='No Post data in Server '+url+' response');
            cb(NO_POST_DATA_REPLY,200,s);
            }
          }
        else
          onError();
          
        xhr=0;
        }
      };

    // Send body as JSON.
    if ( DEBUG )
      logInfo('Sending POST ('+url+'), auth = '+(!!auth)+', data: ',obj);

    try
      {
      xhr.open('POST',url,true);
      xhr.setRequestHeader('X-iizi',_uuid);                    // Device UUID in 'X-iizi' header.
      xhr.setRequestHeader('Content-Type','application/json'); // Default UTF-8.

      // Authentication (only over HTTPS)?
      if ( auth && isSecure )
        {
        if ( _userID && _password )
          xhr.setRequestHeader('Authorization','Basic '+btoa(decodeURIComponent(escape(_userID+':'+_password))));
        else
          throw new Error('Creadentials not set');
        }
 
      xhr.send(JSON.stringify(obj));
      }
    catch(e)
      {
      logErr('Post '+url+' exception',obj,e);
      onError(POST_FAILED,0,'Exception: '+e.message);
      }
    }
    
  /**
   * Function postPromise(code,obj,auth)
   *
   * <p>Posts a Push operation 'code' with the JSON 'obj' to the server
   * at 'host' on 'port' using 'ssl'. The callback 'cb' is invoked with
   * the two parameters: 'error String' (undefined for OK), and JSON {Object}
   * (undefined for error).
   *
   * @type Function
   *
   * @param {String}  func  The push function to perform, appended to the URL
   *                        as "url+'/iizi-push/'+func".
   * @param {Object}  obj   The data Object to post as JSON.
   * @param {Boolean} auth  Truish if Basic authentication credentials
   *                        should be sent or not (only over secure
   *                        connection), 'falsy' otherwise.
   *
   * @return {Promise} The promise for the post operation.
   *                   A rejection will have an {Error} with the
   *                   member 'status' containing the HTTP error
   *                   status. The status value can be zero in case
   *                   the post failed connecting to the host.
   */
  function postPromise(func,obj,auth)
    {
    return new Promise(function(resolve,reject)
      {
      post(func,obj,auth,function(code,status,err,reply)
        {
        if ( code )
          callReject(reject,code,err,{ status: status });
        else
          resolve(reply);
        });
      });
    }
 
  /**
   * Function getServerKeys(appID,force)
   *
   * <p>Gets the server keys used for the 'appID'. The 'appID' is an application ID
   * defined for the current server, or '*' for the Server system (iiziRun) global
   * or 'all apps'. The server keys are cached for a period of 5 minutes in order
   * to reduce network traffic. The return value is a Promise that resolves as
   * a {Boolean} success flag.
   * 
   * @type Function
   *
   * @param {String}  appID  The application ID or '*' for all.
   * @param {Boolean} force  'Boolish' flag to force new keys from server.
   *
   * @return {Promise}  A promise with the server keys Object that contains
   *                    the members 'k' for the keys {Object} as described below
   *                    and a local epoch time {Number} when create was retrieved.
   *
   * <p>The keys {Object} 'k' contains the following members:
   * <ul>
   *   <li>'v' = VAPID public key in Base64 URL encoding
   *   <li>'w' = Firebase Web Configuration {Object}, used to configure a app for
   *             the 'server' and 'appID' in question.  
   * </ul>
   */
  function getServerKeys(appID,force)
    {
    return new Promise(function(resolve,reject)
      {
      // Check that server is defined.
      if ( !server )
        throw new Error('No Server is set');

      // Check if keys are available now.
      var keys=_serverKeys[server];
      if ( !keys )
        keys=_serverKeys[server]={};

      // Get specific keys for app ID.
      if ( !keys[appID] )
        keys[appID]={ t: 0 };
 
      keys=keys[appID];
 
      // Check if the keys are present and not older than 5 minutes.
      if ( !force && keys.k && keys.t+300000<Date.now() ) // 300000 = 5 * 60 * 1000 = 5 minutes.
        resolve(keys.k);
      else
        post('keys',{ app_id: appID },0,function(code,status,err,reply)
          {
          if ( code )
            callReject(reject,code,'Get server keys failed: '+err,{ status: status });
          else
            {
            // Save keys reply and set timestamp.
            keys.k=reply;
            keys.t=Date.now();
            resolve(reply);
            }
          });
      });
    }
    
  /**
   * Function getInvalidServer()
   *
   * <p>Checks if it's localhost (or alike) or is secure (HTTPS). 
   *
   * @type Function
   *
   * @return {String} Error message if not secure or not localhost, otherwise ''.
   */
  function getInvalidServer()
    {
    return (isSecure || isLocalhost)?
      0:
      'Push Server must be secure or "localhost".';
    }
  
  /**
   * Function validateAppTopic(appID,topic)
   *
   * <p>Validates the AppID and the Topic.
   *
   * @type Function
   *
   * @param {String} appID  The App ID.
   * @param {String} topic  The topic.
   *
   * @return {Object} Error message object with members 'c' = error
   *                  code and 'm' = message, or 'falsy' for valid names. 
   */
  function validateAppTopic(appID,topic)
    {
    var err;
    return (appID!='*' && (err=_izRun.validateName(appID)))? {c: INVALID_APPID, m: 'Application ID'+err }:
           (topic      && (err=_izRun.validateName(topic)))? {c: INVALID_TOPIC, m: 'Topic'         +err }: 0;
    }

  /*
   * Function getServer(cbError)
   *
   * <p>Gets the validated server.
   *
   * @type Function
   *
   * @param {Function} cbError The error callback.
   *
   * @return The server URL, or {@code falsy} for error
   *         (and the {@code cbError} callback has been invoked). 
   *
  function getServer(cbError)
    {
    if ( !server )
      cbError(NO_SERVER,'Server is not set');
 
    return server;
    }*/
    
  /**
   * Function validateServerAppTopic(appID,topic)
   *
   * <p>Validates the Server, AppID and the Topic.
   *
   * @type Function
   *
   * @param {String} appID  The App ID.
   * @param {String} topic  The topic.
   *
   * @return {Object} Error message object with members 'c' = error
   *                  code and 'm' = message, or 'falsy' for valid names. 
   */
  function validateServerAppTopic(appID,topic)
    {
    var err;
    return (!server               )? { c: NO_SERVER     , m: 'Server is not set'  }:
           (err=getInvalidServer())? { c: INVALID_SERVER, m: err                  }:
           validateAppTopic(appID,topic);
    }

  /*
   * Function getRunSettings()
   *
   * @type Function
   *
   * <p>Gets the iiziRun push settings object.
   *
   * @return {Object} The 'push' object of the settings, 'falsy' for not available.
   *
  function getRunSettings()
    {
    // If push data is not found: create it.
    var p=_izRun.settings.push;
    if ( !p )
      p=_izRun.settings.push={};
 
    // Returns the 'push' member of the settings if found, otherwise undefined.
    return p;
    }
 
  /**
   * Function flushRunSettings()
   *
   * @type Function
   *
   * <p>Invalidates the iiziRun settings after the object itself has been modified.
   * This will eventually trigger a write-to-storage by iiziRun. If the iiziRun is
   * not found, nothing will happen, nothing is logged. 
   *
  function flushRunSettings()
    {
    // Mark settings as dirty: this will flush it to disk soon.
    _izRun.setDirty();      
    }
*/    
    
  /**
   * Function postMessage(obj)
   *
   * <p>Posts a message on the message to the Service Worker and invokes
   * the 'cbReply' with the 'event.data' reply with the type 'reply'.
   * The rest of the data is up to the transaction parties.
   *
   * @type Function
   *
   * @param {Object}   obj      The {Object} to send (must have the 'type' set).
   * @param {Function} cbError  Optional error callback that will be invoked in case
   *                            the ServiceWorkerController instance is null. In all
   *                            cases, an error will be logged if so. The callback
   *                            takes a {Number} error code and a {String} as
   *                            parameters that will hold the error message.
   *
   * @return {Boolean} 'truish' for success, 'falsy' for failure.
   */
  function postMessage(obj,cbError)
    {
    // This will work when Service Worker is present,
    // even in Safari for the Web (iOS and macOS).
    var c=_serviceWorker && _serviceWorker.controller,s;
    if ( c )
      c.postMessage(obj);
    else
      {
      s='ServiceWorkerController not present';
      logErr(s);
      if ( cbError )
        cbError(NO_SERVICE_WORKER_CONTROLLER,s);
      }
 
    return c;
    }
    
  /*
   * Function postReplyTransaction(obj,cbReply,cbError)
   *
   * <p>Posts a message on the message to the Service Worker and invokes
   * the 'cbReply' with the 'event.data' reply with the type 'reply'.
   * The rest of the data is up to the transaction parties.
   *
   * @type Function
   *
   * @param {Object}   obj      The {Object} to send.
   * @param {Function} cbReply  The reply callback that is invoked with the reply {Object}.
   * @param {Function} cbError  Error callback that will be invoked in case
   *                            the ServiceWorkerController instance is null. In all
   *                            cases, an error will be logged if so. The callback
   *                            takes a {Number} error code and a {String} as
   *                            parameters that will hold the error message.
   *
  function postReplyTransaction(obj,cbReply,cbError)
    {
    // Assign reply callback.
    _cbPostTransaction=function(data)
      {
      // Unassign the MessageChannel.port1 onmessage('reply') callback and invoke reply callback.
      _cbPostTransaction=0;
      cbReply(data);
      };
 
    // Do post message.      
    if ( !postMessage(obj,cbError) )
      _cbPostTransaction=0;
    }
 */
 
  ///
  /// --- window.izPush singleton ---
  ///
  
  window.izPush={
    
  /**
   * Property swVersion
   *
   * <p>The version of the Service Worker or empty string if there is no worker.
   *
   * @type String
   */
  swVersion: '',
 
  /**
   * Function init(cbReg,cbNotify)
   *
   * <p>Initializes the notification and starts listening for any.
   *
   * @type Function
   *
   * @param {Function} cbReg     Callback for registration of service worker completed,
   *                             parameter is {Boolean} for success. Error message is logged. 
   * @param {Function} cbNotify  Callback when notifications are received. A notification
   *                             {Object} is sent, described in the main part of the Push
   *                             Notification documentation (top of the 'push.js' file).
   */
  init: function(cbReg,cbNotify)
    {
    var self=this,d,r,
        settings=_izRun.settings,
        oldUUID=settings.uuid; // Get and save the old UUID.
    
    // Notification object.
    _notification=window.Notification;
   
    // Firebase in Cordova.
    _firebaseCordova=window.FirebasePlugin;
    
    // Firebase for Web.
    _firebaseWeb=!hasCordova && window.firebase;
   
    // Safari Browser under macOS or iOS when not using Cordova.
    _safariWeb=!hasCordova && window.safari;
    
    // Safari Browser push notification instance under macOS or iOS when not using Cordova.
    _safariPush=_safariWeb && _safariWeb.pushNotification;
    
    _uuid=oldUUID;
    
    // Done function that notifies the callback.
    function done(ok)
      {      
      // If OK, see if UUID is set (from Cordova Device ID), generate one.
      if ( !_uuid )
        _uuid=UUID();
 
      // Check for different UUID.
      if ( _uuid!==oldUUID )
        {
        // Save to iiziRun settings and write them to storage (asynchronously).
        settings.uuid=_uuid;
        _izRun.setDirty();
        }
 
      // Add iiziRun Program and Version.
      d=_izRun.PG.replace(' ','_');
      _userAgent+=' '+d+'/'+_izRun.version;
 
      if ( !d.startsWith('iiziRun') )
        d+=' ('+_izRun.AI+', like iiziRun)';

      // Callback success state.
      cbReg(!!ok);
      }
      
    // Handles Safari for the Web (macOS and iOS) without Cordova.
    // This is needed because Safari might not use Service Workers.
    // Only Safari 11+ handles that.
    function initSafari()
      {
      // TODO!
      done(1);
      }
      
    // Check if Cordova-based.
    if ( hasCordova )
      {
      if ( _firebaseCordova )
        {
        // Cordova used for push notifications.
        _firebaseCordova.onMessageReceived(
          function(msg)
            {
            logInfo('Cordova Firebase onMessageReceived',msg);
            if ( msg.messageType=='notification' )
              {
              logInfo('Cordova Firebase notification: message received');
              if ( msg.tap )
                logInfo('Cordova Firebase notification, tapped in',msg.tap);
              }
 
            // Invoke any listener.
            onPush(msg,cbNotify);
            },
          function(err)
            {
            logErr('Cordova Firebase onMessageReceived failed',err);
            });
            
        // Add token refresh support.
        _firebaseCordova.onTokenRefresh(
          function(token)
            {
            logInfo('Cordova Firebase onTokenRefresh',token);
            _cordovaFirebaseToken=token;
            },
          function(err)
            {
            logErr('Cordova Firebase onTokenRefresh failed',err);
            });

         // If called on Android, it gives error messages. 
         if ( (_iOS=(device.platform=='iOS')) )
           {
           // iOS: add new APNS token listener and get it.
           _firebaseCordova.onApnsTokenReceived(
              function(token)
                {
                logInfo('Cordova Firebase onApnsTokenReceived',token);
                _cordovaTokenAPNS=token;
                },
              function(err)
                {
                logErr('Cordova Firebase onApnsTokenReceived failed',err);
                });
              
           _firebaseCordova.getAPNSToken(
              function(token)
                {
                logInfo('Cordova Firebase getAPNSToken',token);
                _cordovaTokenAPNS=token;
                },
              function(err)
                {
                logErr('Cordova Firebase getAPNSToken failed',err);
                });
          }
 
        // Get the Device UUID.
        d=window.device;
        _uuid=window.UUID(d.uuid);
 
        // Build "user agent" similar to a brower's.
        _userAgent='Cordova/'+d.cordova+' ('+d.model+'; '+d.platform+'/'+d.version+' '+d.manufacturer;
        if ( d.isVirtual )
          _userAgent+=' VIRTUAL';

        // When present: according to Cordova documentation, for Android and OSX.          
        if ( (d=d.serial) )
          _userAgent+='; s/n '+d;
 
        _userAgent+=')';

        // Done OK.
        done(1);
        }
      else
        {
        // No Firebase in Cordova!
        logErr('Firebase Cordova not found');
        done(0);
        }
      }
    else
    if ( _notification && _serviceWorker && (_swPush=window.swPush) )
      {
      // Wait for Service Worker: this will later always be OK!
      _serviceWorker.ready.then(function(/*ok*/)
        {
        if ( (d=_serviceWorker.controller) )
          {
          // The 'oncontrollerchange' property of the ServiceWorkerContainer interface is
          // an event handler fired whenever a 'controllerchange' event occurs - when the
          // document's associated ServiceWorkerRegistration acquires a new active worker.
          addEventListener('controllerchange',function(e)
            {
            // TODO? change _serviceWorker.
            logInfo('=====>>> controllerchange event',e);
            });
 
          // Create a message channel.
          _messageChannel=new MessageChannel();
          
          // Listen to incoming messages.
          _messageChannel.port2.onmessage=function(e)
            {
            var d=e.data;
            if ( d )
              switch(d.type)
                {
                // Version sent after 'port' has been posted.
                case 'ver':
                  logInfo('Using ServiceWorker version '+(self.swVersion=d.v));
                  break;
                  
                // Push notification message.
                case 'push':
                  if ( (d=d.p) ) // Grab the 'p' member in the message data from the Service Worker.
                    onPush(d,cbNotify);
                  break;

                // Reply for a postTransaction(..).
                case 'reply':
                  var cb=_cbPostTransaction;
                  if ( cb )
                    {
                    _cbPostTransaction=0;
                    cb(d);
                    }
                  else
                    logErr('No reply function assigned in MessageChannel.port2.onmessage, data',d);
                  break;
                }
            };
          
          // Post port initialize message.
          r=window.izRun;
          d.postMessage(
            {
            type: 'port',
            p   : r.PG,        // Program name of the correct locale selected, e.g. 'iiziRun'.
            i   : r.AI,        // Program ID, e.g. 'com.iizix.run.devel'.
            v   : r.version,   // Version number e.g. '2.0.0'
            t   : r.timestamp  // Timestamp when built, e.g. to clear cache.
            },[_messageChannel.port2]);
 
          // We could have Safari for the Web even with Service Workers! 
          if ( _safariWeb )
            {
            // Safari for the Web (macOS and iOS) without Cordova and Service Worker.
            initSafari();
            }
          else
            {
            // Done OK.
            done(1);
            }
          }
        else
          {
          logErr('ServiceWorkerController is null probably after a Hard Refresh in browser, reloading page');
          location.reload();
          done(0);
          }
        });
      }
    else
    if ( _firebaseWeb && (_firebaseWebMessagingGlobal=_firebaseWeb.messaging()) )
      {
      // Firebase for the Web messaging.
      _firebaseWebMessagingGlobal.getToken(
        function(token)
          {
          // Firebase Web got token.
          logInfo('Firebase Web token (initialize)',token);
          _firebaseWebToken=token;
          },
        function(e)
          {
          logErr('Firebase Web getToken failed: '+e.message);
          done(0);
          });
      }
    else
    if ( _safariWeb )
      {
      // Safari for the Web (macOS and iOS) without Cordova and Service Worker.
      initSafari();
      }
    else
      {
      // No support: not OK.
      done(0);
      }
    },
    
  /**
   * Function info()
   *
   * <p>This method is typically used for CarshLytics.
   *
   * @type Function
   *
   * @return {Object} continting of the following members:
   * <ul>
   *  <li>u:  The UUID, if present, otherwise empty string.
   *  <li>a:  The user agent.
   * </ul>
   */
  info: function()
    {
    return { u: _uuid, a: _userAgent };
    },
    
  /**
   * Function isSupported()
   *
   * <p>Checks if push notifications functionality is supported or not.
   * This DOES NOT CHECK the state of notifications, i.e. if the user has
   * chosen to accept notifications or not, it just verifies if the current
   * environment supports it.
   *
   * @type Function
   *
   * @return {Boolean} true if supported, false otherwise.   
   */
  isSupported: function()
    {
    // Supported in Service Worker or Firebase (in Cordova and also the web).
    return !!(_swPush || _firebaseCordova || _firebaseWeb || _safariWeb);
    },
  
  /**
   * Function state(cbReply,cbError)
   *
   * <p>Gets the current state of notification permissions.
   *
   * @type Function
   *
   * @param {Function} cbReply  Callback with a function that will receive a {String}
   *                            as 1='granted', -1='denied', 0=not unconfigured.
   * @param {Function} cbError  Callback with an error code {Number} as first argument
   *                            and an error message {String} as second parameter.
   */
  state: function(cbReply,cbError)
    {
    getState(
      function(s)
        {
        cbReply(perm(s));
        },
      cbError);
    },
    
  /**
   * Function reqPerm(cbReply,cbError)
   *
   * <p>Requests push notification permissions to be enabled if it is not.
   *
   * @type Function
   *
   * @param {Function} cbReply  Callback function that will be the new state as
   *                            1=Granted, 0=Default (canceled or unconfigured),
   *                            -1=Denied or -2=Not available (unsupported).
   * @param {Function} cbError  Error callback with {Number} error code and {String}
   *                            with error message. 
   */
  reqPerm: function(cbReply,cbError)
    {
    // Get state.
    var s;
    this.state(
      function(state)
        {
        if ( state )
          {
          // 1=granted or -1=denied.
          cbReply(state);
          }
        else
          {
          // 0=Unconfigured or Default.
          if ( _firebaseCordova )
            {
            // Firebase with Cordova.
            _firebaseCordova.grantPermission(
              function(ok)
                {
                // Get the App ID for Firebase.
                _firebaseCordova.getId(
                  function(id)
                    {
                    _cordovaFirebaseID=id;
                    log('Cordova Firebase app ID = '+id);
   
                    // It's either granted or denied, not "default".
                    cbReply(ok? GRANTED: DENIED);
                    },
                  function(err)
                    {
                    logErr(s='Cordova Firebase getId failed: '+err);
                    cbError(GET_FIREBASE_ID_FAILED,s);
                    });
                },
              function(err)
                {
                logErr(s='Cordova Firebase grantPermission failed: '+err);
                cbError(GRANT_FIREBASE_PERMISSION_FAILED,s);
                });
            }
          else
          if ( _safariWeb )
            {
            // Safari Web.
            if ( _safariPush )
              {
              // Server must be set.
              if ( _safariWebsitePushID )
                {
                // User/password must be set.
                if ( _userID && _password )
                  {
                  _safariPush.requestPermission(server,_safariWebsitePushID,
                    { uuid: _uuid, user: _userID, epw : _password },
                    function(p)
                      {
                      var s=p.permission;
                      logInfo('Safari Web requestPermission',_safariWebsitePushID,s);
                      cbReply(perm(s));
                      });
                  }
                else
                  {
                  logErr(s='Safari Web request permission requires User to be set');
                  cbError(NO_USER,s);
                  }
                }
              else
                {
                // Return 'unconfigured'.
                logErr('Safari Web needs Secure Server to be set');
                cbError(NO_SERVER,s);
                }
              }
            else
              {
              logErr('Safari Web does not support Push Notifications API');
              cbReply(UNSUPPORTED);
              }
            }
          else
          if ( _firebaseWeb )
            {
            // Firebase for the Web.
            }
          else
          if ( _swPush )
            {
            // Service Worker default state: we can request permission.
            _notification.requestPermission()
              .then(function(p)
                {
                // Callback with our state value 1 to -1.
                logInfo('Service Worker requestPermission = '+p);
                cbReply(perm(p));
                })
              .catch(function(err)
                {
                logErr(s='Failed Service Worker requestPermission',err);
                cbError(REQUEST_VAPID_PERMISSION_FAILED,s+': '+err.message);
                });
            }
          else
            {
            // No providers: unsupported.
            cbReply(UNSUPPORTED);
            }
          }
        },
      cbError);
    },
    
  /**
   * Sets the server host, port and SSL (secure) flag that shall be used for subsequent
   * push notification operations. Please note that push notification operations only
   * will work if the server is secure or the host is 'localhost' (or similar, e.g. '127.0.0.1').
   * When setting the server, it's server keys are invalidated. 
   *
   * @type Function
   *
   * @param {String}  host    The host name or address, undefined or null for none.
   * @param {Number}  port    The port number.
   * @param {Boolean} secure  The secured SSL flag (use 'https' or 'http').
   *
   * @return {String} Server URL as 'http[s]://port[:port]' with 'http' or 'https'
   *                  depending on 'secure' flag, and ':port' only appended if it is
   *                  not the default port (80 or 443). Empty string is returned if 'host'
   *                  is 'falsy'. 
   */
  setServer: function(host,port,secure)
    {
    if ( host )
      {
      host=host.toLowerCase();
      if ( (isSecure=secure) )
        {
        server='https://'+host;
        if ( port!=443 )
          server+=':'+port;
        
        // Always HTTPS: Safari Website Push ID is 'web.' + reversed domain name (minimum 2 groups, e.g. 'example.com').
        // Verify that the host name is valid.
        _safariWebsitePushID=(host.indexOf('.')<0 || _izRun.validateName(host))?
           '':
           'web.'+host.split('.').reverse().join('.');
        }
      else
        {
        // Fast RegEx not caring about validating the IP v4 and v6 address (e.g. it could be 127.9 or ::0::::1).
        isLocalhost=host.match(/^(localhost|127\.[\d.]+|[0:]+1)$/);
        server='http://'+host;
        if ( port!=80 )
          server+=':'+port;
 
        // Never HTTP for Safari Web Push.
        _safariWebsitePushID=0;
        }
  
      // Brand new keys!
      if ( !_serverKeys )
        _serverKeys={};
       
      _serverKeys[server]={};
      }
    else
      {
      // A previous server will clear the keys of it.
      if ( server && _serverKeys )
        delete _serverKeys[server];

      // Clear the server variables.
      isSecure=isLocalhost=server=_safariWebsitePushID=undefined;
      }
 
    return server;
    },
    
  /**
   * Function setUser(userID,epw)
   *
   * <p>Sets the user ID and password for Server authentication.
   * The user ID MUST be set AFTER the server has been set, otherwise an Error is thrown.
   *
   * <p><b>N O T E </b>: If the 'userID' or 'epw' is 'falsy', an Error is thrown. If both are 'falsy', the user is cleared.
   *
   * @type Function
   *
   * @param {String} userID  User ID.
   * @param {String} epw     Encrypted user's password of clear text or a hashed version from the server.
   */
  setUser: function(userID,epw)
    {
    // If either is set, check that 'userID' and 'epw' are set.
    if ( userID || epw )
      {
      // Both must be set.
      if ( !userID || !epw )
        throw new Error('User credentials unset');
 
      _userID=userID;
      _password=epw;
      
      // Firebase Analytics for the Web?
      var a=window.fbwAnalytics;
      if ( a && _izRun.FB_ANALYTICS )
        a.setUserId(userID);
   
      // Pass this along to the service worker, if any.
      if ( _swPush )
        {
        // Pass the user ID and password to the worker who replies with the UUID of the user.
        postMessage(
          {
          type: 'user',
          data:
            {
            // Key values.
            url  : server,
            user : userID,
  
            // Additional data.
            epw  : epw,
            uuid : _uuid,
            agent: _userAgent
            }
          });
        }
      else
      if ( _firebaseCordova )
        {
        // In Firebase Cordova, set the user ID for Analytics.
        _firebaseCordova.setUserId(userID);
        _firebaseCordova.setCrashlyticsUserId(userID);
        }
      }
    else
      {
      // Clear user/password by setting to 'falsy'.
      _userID=_password=0;
      }
    },

  /*
   * Function flush()
   *
   * <p>Informs a potential Service Worker that it can should write the settings
   * data and flush it to cache as soon as possible.
   *
   * @type Function
   *
  flush: function() 
    {
    if ( _swPush )
      postMessage(
        {
        type: 'flush'
        });
    },*/
    
  /**
   * Function getSubscriptionCount(cbReply,cbError)
   *
   * <p>Gets the current count of subscriptions. If there is no server to request a reply from,
   * just check if there are any current notifications available.
   *
   * @type Function
   *
   * @param {Function} cbReply  Callback that has one {Number} parameter with the count
   *                            of currently active subscriptions.
   * @param {Function} cbError  Callback with a {Number} return code and a error message
   *                            {String} in case this fails.
   */
  getSubscriptionCount: function(cbReply,cbError)
    {
    var s;
    if ( server )
      {
      // Make sure user and password are set.
      if ( !_userID || !_password )
        cbError(NO_USER,'No User/Password has been set');
      else
        {
        // Get the count of subscriptions from server.
        log('Safari Web not yet supported: no subscriptions');
        this.getAllSubscriptions('*',function(subs)
          {
          cbReply(subs.length);
          },cbError);
        }
      }
    else // Fallback to check locally for any subscription.
    if ( _safariWeb )
      {
      // Safari for the Web (iOS and macOS without Cordova).
      // TODO!
      log('Safari Web not yet supported: no subscriptions');
      cbReply(0);
      }
    else
    if ( _swPush )
      {
      // Just get the current subscription.
      _swPush.pushManager.getSubscription()
        .then(function(sub)
          {
          if ( sub )
            {
            // There is a subscription present: save it (but we don't know what app!)
            //_vapidSubscriptions=_vapidSubscriptions||{};
            //_vapidSubscriptions[appID]=sub;
            log('getSubscriptionCount: present (one)');
            cbReply(1);
            }
          else
            {
            // No VAPID subscriptions.
            log('getSubscriptionCount: none');
            cbReply(0);
            }
          })
        .catch(function(err)
          {
          s=err.message;
          logErr('Failed ServiceWorker.pushManager.getSubscription(): '+s);
          cbReply(GET_VAPID_SUBSCRIPTION_FAILED,s);
          });
      }
    else
      {
      // Firebase Cordova can't get current subscription.
      // Firebase Web can't get current subscription.
      // No support.
      // Return no subscriptions.
      cbReply(0);      
      }
    },
 
  /**
   * Function getAllSubscriptions(appID,cbReply,cbError)
   *
   * <p>Gets all subscriptions from the server for current user for all devices.
   *
   * @type Function
   *
   * @param {String}   appID       The application ID, or '*' for all.
   * @param {Function} cbReply     Callback that has one {Array} parameter with the will
   *                               receive the active subscriptions with the members as below.
   * @param {Function} cbError     Callback with a {Number} return code and a error message
   *                               {String} in case this fails.
   *
   * + TODO...
   */
  getAllSubscriptions: function(appID,cbReply,cbError)
    {
    // Validate server and app ID.
    var s=validateServerAppTopic(appID,0);
    if ( s )
      cbError(s.c,s.m);
    else
      {
      // Ask server.
      /*
      push('n',{ d: false },1,function(code,status,err,data)
        {
        if ( code )
          {
          var s=s='Failed retrieving subscription count from server '+server;
          logErr(s,code,err);
          cbError(code,s);
          }
        else
          {
          logInfo('getSubscriptionCount: user = '+_userID+', appID = '+appID+', topic = '+topic+': count = '+data.c);
          cbReply(data.c);
          }
        });*/
        
      // For now... empty array...
      cbReply([]);
      }
    },
    
  /**
   * Function reset(cb)
   *
   * <p>Resets the Push notifications and a potential Service Worker.
   * All subscriptions of the user on THIS device will be unsubscribed.
   * For Cordova, this will also clear all pending notifications from the drawer.
   *
   * @type Function
   *
   * @param {Function} cb  Callback that has a return code {Number} and a {String}
   *                       parameter with error message that is undefined when successful.
   */
  reset: function(cb)
    {
    function onErr(err)
      {
      logErr('failed reset(): ',err);
      cb(DENIED,(err.message || err));
      }
      
    if ( _firebaseCordova )
      {
      // TODO: Get the list of open notifications so that server can be informed!
      _firebaseCordova.clearAllNotifications();
      _firebaseCordova.unregister();
      cb(SUCCESS,0);
      }
    else
    if ( (_firebaseWeb/*=...*/) )
      {
      // TODO: Firebase for the Web.
      cb(SUCCESS,0); // For now...
      }
    else
    if ( (_safariWeb/*=...*/) )
      {
      // TODO: Safari for the Web (macOS and iOS) without Cordova.
      cb(SUCCESS,0); // For now...
      }
    else
    if ( _swPush )
      {
      // Just get the current subscription.
      _swPush.pushManager.getSubscription()
        .then(function(sub)
          {
          if ( sub )
            {
            // There is a subscription present: save it (but we don't know what app!)
            //_vapidSubscriptions=_vapidSubscriptions||{};
            //_vapidSubscriptions[appID]=sub;
            log('Reset: unsubscribe: '+sub);
            cb(1);
            }
          else
            {
            // No VAPID subscriptions.
            log('Reset: VAPID - getSubscriptionCount: none');
            cb(SUCCESS,0);
            }
          })
        .catch(onErr);
      }
    else
      {
      // Not OK.
      cb(UNSUPPORTED,'Push notifications are not supported');
      }
    },
    
  /**
   * Function getSubscriptions(appID,topic,cbReply,cbError)
   *
   * <p>Gets the subscriptions available for the 'appID' with 'topic'.
   *
   * @type Function
   *
   * @param {String}   appID    The application ID, or '*' for all.
   * @param {String}   topic    Topic to use or empty for none.
   * @param {Function} cbReply  Callback with two parameters as described below.
   * @param {Function} cbError  Callback with a {Number} return code and a error message
   *                            {String} in case this fails.
   *
   * <ol>
   *   <li>First a {Number} of subscriptions: there can be multiple subscriptions
   *       from several devices for the 'current user', 'appID' and 'topic'.
   *   <li>Second, an {Object} that might be 'falsy' if there is no current
   *       subscription for this 'appID' and 'topic', otherwise as described below:
   * </ol>
   *
   * <ul>
   *   <li>'c' = {Number} Count of subscriptions: zero or one,
   *   <li>'a' = {String} App ID,
   *   <li>'t' = {String} Topic,
   *   <li>'k' = {Object} Server keys used for the subscription, and
   *   <li>'s' = {Object} subscriptionObject (see below).
   * </ul>
   *
   * <p>The subscriptionObject is an {Object} with the following members for the
   * provider that supported the subscription, and it contains the provider-specific
   * {Object}, e.g. {PushSubscription} for VAPID. This {Object} is not set if there
   * are no current subscriptions currently available. The possible providers are:
   *
   * <ul>
   *   <li>'v' = {PushSubscription} VAPID-based provider using Service Worker,
   *   <li>'a' = {String} Firebase Cloud Messaging in Cordova for Android,
   *   <li>'i' = {String} Firebase Cloud Messaging in Cordova for iOS,
   *   <li>'w' = {String} Firebase Cloud Messaging of the Web,
   *   <li>'s' = {String} Apple Push in Safari (could be macOS and iOS without Cordova).
   * </ul>
   */
  getSubscriptions: function(appID,topic,cbReply,cbError)
    {
    // Validate server, 'appID' and 'topic'.
    var s=validateServerAppTopic(appID,topic);
    if ( s )
      {
      var code=s.c,err=s.m;
      logErr(s='Failed getting subscriptions: '+code,err);  
      cbError(code,s);
      }
    else
      {
      // Ask server for the subscriptions.
      post('get_subs',{ a: appID, t: topic },1,
        function(code,status,err,data)
          {
          if ( code )
            {
            var txt='Failed getting subscriptions from server: '+err;
            logErr(txt,code,status);
            cbError(code,txt);
            }
          else
            {
            // Show subscription data JSON.
            var subs=0,count=0;
            if ( DEBUG )
              {
              if ( data )
                log('Server replied with subscriptions = '+JSON.stringify(data,null,2));
              else
                log('No subscriptions data returned from server');
              }
 
            // Add the providers current subscription.
            if ( _firebaseCordova )
              subs=(_cordovaTokenAPNS    )? { i: _firebaseCordovaSubscription }:
                   (_cordovaFirebaseToken)? { c: _firebaseCordovaSubscription }: 0;
            else
            if ( _firebaseWebMessagingGlobal )
              {
              // TODO: global messaging objects...
              // Return nothing for now.
              // subs=0;
              /*getFirebaseWebMessaging                
              s={ w: _firebaseWebSubscription };*/
              }
            else
            if ( _safariPush )
              subs={ s: _safariSubscription };
            else
            if ( _swPush )
              {
              // Get the subscription for the Service Worker.
              _swPush.pushManager.getSubscription()
                .then(function(sub)
                  {
                  if ( sub )
                    cbReply(1,{v: sub});
                  else
                    cbReply(0);
                  })
                .catch(function(e)
                  {
                  var txt='Failed getting current VAPID subscription';
                  logErr(txt,e);
                  cbError(GET_VAPID_SUBSCRIPTION_FAILED,txt+': '+e.message);
                  });
                
              return;
              }
              
            // Count!
            if ( subs )
              ++count;

            // Reply with current subscriptions.
            cbReply(count,subs);
            }
          });
      }
    },
 
  /**
   * Function sendPush(appID,topic,msgObj,cb)
   *
   * @type Function
   *
   * @param {String}   appID    Application ID, '*' for All apps.
   * @param {String}   topic    Topic, empty string for any (all).
   * @param {String}   msgObj   A message {Object} as described below.
   * @param {Array}    dests    Destinations (with user ID's or their user's email address).
   * @param {Function} cbReply  Callback with {Number} of destination sent to and a {String}
   *                            parameter for message.
   * @param {Function} cbError  Callback with a {Number} return code and a error message
   *                            {String} in case this fails.
   */
  sendPush: function(appID,topic,msgObj,dests,cbReply,cbError)
    {
    // Verify that 'appID' and 'topic' are both valid.
    var s=validateServerAppTopic(appID,topic);
    if ( s )
      cbError(s.c,s.m);
    else
      post('push',{ a: appID, t: topic, m: msgObj, d: dests },1, // 'p' = send Push notification (authenticated).
        function(code,msg,status,reply)
          {
          if ( code )
            {
            logErr('failed sending push message',code,msg);
            cbError(code,msg);
            }
          else
            {
            logInfo('Successfully sent push message, reply = ',reply);
            if ( reply && reply.c )
              cbReply(reply.c,reply.c==1?
                'Successfully sent push message to one recipient.':
                'Successfully sent push message to '+reply.c+' recipients.');
            else
              cbReply(0,'Successfully sent push message, but there were no recipients.');
            }
          });
    },
  
  /**
   * Function subscribe(appID,topic,cb)
   *
   * <p>Issues a subscription for notification.
   *
   * @type Function
   *
   * @param {String}   appID  Application ID, '*' for All apps.
   * @param {String}   topic  Topic, empty string for any (all).
   * @param {Function} cb     Callback with two parameters, first error (or null for no
   *                          error), second is a subscriptions {Object}:
   *
   * <ul>
   *   <li>'c'  = Count of subscriptions: zero or one,
   *   <li>'a'  = App ID,
   *   <li>'t'  = Topic,
   *   <li>'u'  = Device UUID,
   *   <li>'U'  = User Agent (Cordova and Browser),
   *   <li>'d'  = Current date in epoch milliseconds,
   *   <li>'k'  = Server keys used for the subscription, and
   *   <li>'s'  = subscriptionObject (see below).
   * </ul>
   *
   * <p>The subscriptionObject is an {Object} with the following members for the
   * provider that supported the subscription, and it contains the provider-specific
   * {Object}, e.g. {PushSubscription} for VAPID. The possible providers are:
   *
   * <ul>
   *   <li>'v'  = VAPID-based provider using Service Worker,
   *   <li>'fa' = Firebase Cloud Messaging in Cordova for Android,
   *   <li>'fi' = Firebase Cloud Messaging in Cordova for iOS,
   *   <li>'fw' = Firebase Cloud Messaging of the Web,
   *   <li>'a'  = Apple Push in Safari (could be macOS and iOS without Cordova).
   * </ul>
   */
  subscribe: function(appID,topic,cb)
    {
    // Error callback.
    function onErr(err)
      {
      console.warn('Failed Push subscribe()',err);
      cb('Subscribe failed: '+(err.message || err)+'.');
      }
      
    // Registers subscriptions object with server.
    function regSrv(srvSubs,orgSubs)
      {
      postPromise('subscribe',srvSubs,1) // 'subscribe' = register subscription (authenticated).
        .then(function(rc)
          {
          console.warn('Registered subscription with server '+server+' = ',rc);
          cb((rc && rc.ok)?
            0:
            'Subscription successful, but was already registered in the Server '+server+'.',
            orgSubs);
          })
        .catch(onErr);
      }
      
    // Verify that 'appID' and 'topic' are both valid.
    var msg=validateAppTopic(appID,topic);
    if ( msg )
      cb(msg);
    else
      {
 
      // TODO: Save subscription in settings or pass to the Service Worker.
 
      var subs={ c: 1, a: appID, t: topic, u: _uuid, U: _userAgent, d: Date.now(), s: {} },
          srvSubs=_mixin({},subs);

      if ( _swPush )
        {
        // Subscribe with the Service Worker.
        getServerKeys(appID)
          .then(function(keys)
            {
            var vapid=keys.v;
            if ( !vapid )
              throw new Error('No VAPID keys present');
  
            _swPush.pushManager.subscribe(
              {
              userVisibleOnly: true,
              applicationServerKey: vapid
              })
              .then(function(sub)
                {
                // Send back 'a'=appID, 't'=topic, 's'=subscription, etc...
                console.warn('Got Service Worker subscription = ',sub);
                if ( sub )
                  {
                  // Cache subscription.
                  _vapidSubscriptions=_vapidSubscriptions||{};
                  _vapidSubscriptions[appID]=sub;

                  // Register with the server.
                  srvSubs.k=subs.k=keys;
                  subs   .s={ v: sub };
                  srvSubs.s={ v: sub.toJSON() };
                  regSrv(srvSubs,subs);
                  }
                else
                  onErr('VAPID subscribe failed, Server '+server);
                })
              .catch(onErr);
            })
          .catch(onErr);
        }
      else
      if ( _firebaseCordova )
        {
        // Check if there is a token present or not!
        if ( ((_iOS)? _cordovaTokenAPNS: _cordovaFirebaseToken) )
          {
          // Firebase in Cordova: get server keys.
          getServerKeys(appID)
            .then(function(keys)
              {
              // Check for valid for iOS or Android.
              var err;
              if ( _iOS )
                {
                if ( !keys.i )
                  err='Firebase APNS';
                }
              else
              if ( !keys.fc )
                err='Firebase';
  
              if ( err )
                onErr(err+' key is not present');
              else
                {
                // Assign existing keys.
                subs.k=keys;
    
                _firebaseCordova.subscribe(topic,
                  function()
                    {
                    // Handle iOS and Android differently, then register with server.
                    log('Cordova Firebase subscribe success');
                    subs.k=srvSubs.k=keys;
                    subs.s=(_iOS)? { i: _cordovaTokenAPNS }: { fc: _cordovaFirebaseToken };
                    regSrv(subs,subs);
                    },onErr);
                }
              })
            .catch(onErr);
          }
        else
          onErr('The Firebase Token for this device is not yet set.');
        }
      else
        onErr('No support for notification subscriptions');
      }
    },
    
  /**
   * Function unsubscribe(subs,cb)
   *
   * <p>Issues a subscription for notification.
   *
   * @type Function
   *
   * @param {Object}   subs   Subscriptions Object from 'getSubscriptions(...)'.
   * @param {Function} cb     Callback with error message {String} that is undefined
   *                          when successful.
   */
  unsubscribe: function(subs,cb)
    {
    // Error callback.
    function onErr(code,err)
      {
      logErr('Failed Push unsubscribe()',err);
      cb(code,(err.message || err));
      }
      
    // Unsubscribe with the server.
    function unsubscribeServer(srvSubs,type)
      {
      postPromise('unsubscribe',srvSubs,1) // 'unsubscribe' = Cancel subscription (authenticated).
        .then(function(successObj)
          {
          cb((successObj && successObj.ok)?
            0:
            type+' unsubscribe successful, but the Server '+server+' did not have any matching registered subscriptions.');
          })
        .catch(onErr);
      }
 
    // TODO: Save subscription in settings or pass to the Service Worker.
 
    // No subscriptions object means failure.
    var s,sp;
    if ( !subs.c || !(s=subs.s) )
      onErr('No subscription(s)');
    else
    if ( _swPush && (sp=s.v) )
      {
      // Unsubscribe with the Service Worker ('v' = VAPID).
      var json=sp.toJSON(); // For the server.
      sp.unsubscribe()
        .then(function(success)
          {
          // Clear VAPID cached subscription.
          // TODO: All appID's are cleared!
          _vapidSubscriptions=0;
          
          console.warn('Service Worker unsubscribe, success = ',success);
          if ( success )
            {
            // Replace subscription object with to JSON.
            var srvSubs=_mixin({},subs);
            srvSubs.s={v:json};  // Set JSON Object for server.
            unsubscribeServer(srvSubs,'VAPID');
            }
          else
            cb('VAPID unsubscribe failed, Server '+server+'.');
          })
        .catch(onErr);
      }
    else
    if ( _firebaseCordova )
      {
      // Firebase Cordova.
      _firebaseCordova.unsubscribe(subs.t,
        function()
          {
          console.warn('Firebase unsubscribed topic "'+subs.t+'" successfully');
          var srvSubs=_mixin({},subs),s=subs.s;
          if ( _iOS )
            {
            srvSubs.s={ i: _cordovaTokenAPNS };
            if ( s )
              delete s.i;
            }
          else
            {
            srvSubs.s={ fc: _cordovaFirebaseToken };
            if ( s )
              delete s.fc;
            }

          // Unsubscribe in server.
          unsubscribeServer(srvSubs,'Firebase');
          },
        function(err)
          {
          cb('Unsubscribe from topic "'+subs.t+'" failed: '+err+'.');
          });
      }
    else
      {
      // TODO: Add Apple macOS Safari Browser...
      onErr('No support for notification subscriptions');
      }
    }
 
// End of "izPush" singleton.
};

// End of singleton, now start it...
})(window,navigator);
