--------------------------------------------------------------------------------------------------------------


-- USERS

-- The users with access to the server and its apps and other functions such as push notifications
-- subscriptions or the API to send a push notification.

CREATE TABLE IZS.USERS (
 uid              BIGINT       NOT NULL PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY, -- User ID
 user_name_lower  VARCHAR(254) NOT NULL              ,  -- User name in (English) lower case
 user_name        VARCHAR(254) NOT NULL              ,  -- User name as originally entered
 hash_pw          VARCHAR(120) NOT NULL              ,  -- The password hash (PBKDF2 with Keyed-Hash Message Authentication Code (HMAC) with a SHA-256 hash)
 change_pw        BOOLEAN      NOT NULL DEFAULT FALSE,  -- Flag indicating user must change password at next login
 email            VARCHAR(254) DEFAULT NULL          ,  -- User's email address
 lang_codes       VARCHAR(200) DEFAULT NULL          ,  -- Comma separated list of desired BCP47 language codes
 created          BIGINT       NOT NULL              ,  -- Date in epoch milliseconds when user was created
 last_login       BIGINT       NOT NULL DEFAULT 0    ,  -- Last login of user in epoch milliseconds
 last_fail        BIGINT       NOT NULL DEFAULT 0    ,  -- Last failed login of user in epoch milliseconds
 fail_count       SMALLINT     NOT NULL DEFAULT 0    ,  -- Count of failed logins
 disabled         VARCHAR(40)  DEFAULT NULL          ,  -- Reason for disabled user, NULL if user account is enabled
 UNIQUE (user_name_lower)
);


-- USER_DETAILS

-- Details for users, contains additional user information such as phone number(s), alternate mail
-- address(es), profile images, etc.

CREATE TABLE IZS.USER_DETAILS (
 uid     BIGINT        NOT NULL,  -- User ID
 detail  VARCHAR(40)   NOT NULL,  -- Detail name
 value   VARCHAR(3000) NOT NULL,  -- Detail value
 UNIQUE (uid, detail),
 FOREIGN KEY (uid) REFERENCES IZS.USERS(uid) ON DELETE CASCADE
);


-- USER_PASSWORDS

-- History of last used user passwords.

CREATE TABLE IZS.USER_PASSWORDS (
 uid      BIGINT       NOT NULL,  -- User ID
 created  BIGINT       NOT NULL,  -- Date in epoch milliseconds when password was created
 hash_pw  VARCHAR(120) NOT NULL,  -- The password hash (PBKDF2 with Keyed-Hash Message Authentication Code (HMAC) with a SHA-256 hash)
 UNIQUE (uid, created),
 FOREIGN KEY (uid) REFERENCES IZS.USERS(uid) ON DELETE CASCADE
);

CREATE INDEX user_passwords_hash_pw ON IZS.USER_PASSWORDS(uid, hash_pw);


--------------------------------------------------------------------------------------------------------------


-- GROUPS

-- 'gid'        must is unique key
-- 'name_lower' must be unique
-- 'name'       must be unique

FLAVOR_DERBY: -- For Derby!
CREATE TABLE IZS.GROUPS (
 gid        BIGINT       NOT NULL PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY (START WITH 4), -- Group ID
 name_lower VARCHAR(253) NOT NULL              ,  -- Group name in (English) lower case
 name       VARCHAR(253) NOT NULL              ,  -- Original group name with character case
 descr      VARCHAR(160) NOT NULL              ,  -- Group description
 disabled   BOOLEAN      NOT NULL DEFAULT FALSE,  -- Flag if group is disabled (access permission)
 all_apps   BOOLEAN      NOT NULL DEFAULT FALSE,  -- Flag indicating access allowed to all apps
 UNIQUE (name_lower)
);

-- MySQL (START WITH 4) is not possible, use ALTER TABLE tablename AUTO_INCREMENT=4 instead! 
FLAVOR_MYSQL: -- For MySQL!
CREATE TABLE IZS.GROUPS (
 gid        BIGINT       NOT NULL PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY, -- Group ID
 name_lower VARCHAR(253) NOT NULL              ,  -- Group name in (English) lower case
 name       VARCHAR(253) NOT NULL              ,  -- Original group name with character case
 descr      VARCHAR(160) NOT NULL              ,  -- Group description
 disabled   BOOLEAN      NOT NULL DEFAULT FALSE,  -- Flag if group is disabled (access permission)
 all_apps   BOOLEAN      NOT NULL DEFAULT FALSE,  -- Flag indicating access allowed to all apps
 UNIQUE (name_lower)
);

-- MySQL AUTO_INCREMENT START.
FLAVOR_MYSQL: -- For MySQL!
ALTER TABLE IZS.GROUPS AUTO_INCREMENT=4;


-- USER GROUPS

-- Table used to add usera to groups.
-- 'uid','gid' pair must be unique.

CREATE TABLE IZS.USER_GROUPS (
 uid  BIGINT NOT NULL,  -- User ID
 gid  BIGINT NOT NULL,  -- Group ID
 FOREIGN KEY (uid) REFERENCES IZS.USERS (uid) ON DELETE CASCADE,
 FOREIGN KEY (gid) REFERENCES IZS.GROUPS(gid) ON DELETE CASCADE
);

CREATE        INDEX user_groups_uid     ON IZS.USER_GROUPS(uid);
CREATE UNIQUE INDEX user_groups_uid_gid ON IZS.USER_GROUPS(uid, gid);


-- Add groups and users:

-- User name             Password      --> Combined password hash, cannot decrypt it (only hashing)

-- iiziAdmin             secret        --> IZ1:20000:aLFAecQqclZV/XADHKIHIKuBlnVcYM/E5poa7GNgfLI=:1b/yy189I37RysLUuHP019wBR2Nqm5OxW+sN5JCL58Y=
-- iiziAdmin BACKUP      secret BACKUP --> IZ1:20000:Z5/rQ7ZeZ6dpN1iSjW3HAjWgEwuqHT9khwIZAkqUEE4=:EWlbepW+CFVVJofARiMOOEG7Kko8MTzTnaCXxXNlcxI=
-- user                  secret        --> IZ1:20000:AtgIMVAYSyh1PeSEHNZABhwl29WAQgYDMqwyCEq1DjA=:nZlf1wZjof6jnyovfUj8dc83jH96m6LSbOX+C3p9/Rg=
-- iiziRun Developer     secret        --> IZ1:20000:j51LDQxEsZKt0R9KuTUpMUuFq1Qead2xSh4WDabcj74=:8WIVd9JqNSNDye24lxlsI6X01BK9VE4WDr/lXN0RXsc=
-- iiziRun Developer 2   secret        --> IZ1:20000:0Od1/VJR8JN476+06Q3H509ssq99TH+76TJfUKJAEjY=:ZaryioKzuh5ephhykk59XCBnTvhxnHwdw7CmiSQ5FJI=

INSERT INTO IZS.GROUPS VALUES (1,'administrators'              ,'Administrators'              ,'Group for administrators, all apps enabled',FALSE,TRUE );
INSERT INTO IZS.GROUPS VALUES (2,'default'                     ,'Default'                     ,'Default users'                             ,FALSE,FALSE);
INSERT INTO IZS.GROUPS VALUES (3,'default all apps (disabled!)','Default All Apps (disabled!)','Default users with all apps enabled!'      ,FALSE,TRUE );

INSERT INTO IZS.USERS (user_name,user_name_lower,created,email,hash_pw) VALUES
  ('iiziAdmin'          ,'iiziadmin'          ,0,'default@iizi.server.administrator'         ,'IZ1:20000:aLFAecQqclZV/XADHKIHIKuBlnVcYM/E5poa7GNgfLI=:1b/yy189I37RysLUuHP019wBR2Nqm5OxW+sN5JCL58Y='),
  ('iiziAdmin BACKUP'   ,'iiziadmin backup'   ,0,'default@iizi.server.administrator'         ,'IZ1:20000:Z5/rQ7ZeZ6dpN1iSjW3HAjWgEwuqHT9khwIZAkqUEE4=:EWlbepW+CFVVJofARiMOOEG7Kko8MTzTnaCXxXNlcxI='),
  ('user'               ,'user'               ,0,'default@iizi.server.user'                  ,'IZ1:20000:AtgIMVAYSyh1PeSEHNZABhwl29WAQgYDMqwyCEq1DjA=:nZlf1wZjof6jnyovfUj8dc83jH96m6LSbOX+C3p9/Rg='),
  ('iiziRun Developer'  ,'iizirun developer'  ,0,'developers.using.iizi.run@all.apps.enabled','IZ1:20000:j51LDQxEsZKt0R9KuTUpMUuFq1Qead2xSh4WDabcj74=:8WIVd9JqNSNDye24lxlsI6X01BK9VE4WDr/lXN0RXsc='),
  ('iiziRun Developer 2','iizirun developer 2',0,'developers.using.iizi.run@all.apps.enabled','IZ1:20000:0Od1/VJR8JN476+06Q3H509ssq99TH+76TJfUKJAEjY=:ZaryioKzuh5ephhykk59XCBnTvhxnHwdw7CmiSQ5FJI=')
;

INSERT INTO IZS.USER_GROUPS (uid, gid) VALUES
  (1, 1), (1, 2), (1, 3), -- 'iiziAdmin'           to 'Administators', 'Default' and 'Default All Apps!'
  (2, 1), (2, 2), (2, 3), -- 'iiziAdmin Backup'    to 'Administators', 'Default' and 'Default All Apps!'
  (3, 2),                 -- 'user'                to 'Default'
  (4, 2), (4, 3),         -- 'iiziRun Developer'   to 'Default' and 'Default All Apps!'
  (5, 2), (5, 3)          -- 'iiziRun Developer 2' to 'Default' and 'Default All Apps!'
;


--------------------------------------------------------------------------------------------------------------


-- APPS

-- Definition of App ID's. The 'app_id' is in lower case and has the same kind of validation as
-- non-international Internet domains, and in our case we do not check the TLD, just the length being
-- between 2 and 6 characters (a-z) without digits (0-9) or dashes (-).

CREATE TABLE IZS.APPS (
 aid          BIGINT       NOT NULL PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY, -- App numeric ID (unique per app_name_id)
 app_id       VARCHAR(32)  NOT NULL      ,  -- App ID consisting of 32 hexadecimal characters (0-9, a-f), in groups of 4, separated by a dash ('-').
 created      BIGINT       NOT NULL      ,  -- Creation or last modification date of app entry, milliseconds since epoch in GMT, zero for system app
 enabled      BOOLEAN      DEFAULT TRUE  ,  -- Flag for app enabled for use, FALSE if this app instance is disabled
 auto_enroll  BOOLEAN      DEFAULT TRUE  ,  -- Flag to auto-enroll new users or if FALSE, the process is handled by other means (manual or programmatic)
 enroll_email VARCHAR(254) DEFAULT NULL  ,  -- Email to notify when user wishes to enroll to application, NULL for none
 def_version  VARCHAR(64)  NOT     NULL  ,  -- Default version of app to use, typically 1.2.3.45678 (where 45678 is the build number)
 UNIQUE(app_id)
);


-- APP_INSTANCE

-- An instance of an App ID with a specific version and server environment required.

CREATE TABLE IZS.APP_INSTANCE (
 avid         BIGINT       NOT     NULL PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY, -- App numeric unique ID specific to 'version'
 aid          BIGINT       NOT     NULL        ,  -- App numeric ID
 version      VARCHAR(64)  NOT     NULL        ,  -- Version, typically 1.2.3.45678 (where 45678 is the build number)
 environment  VARCHAR(253) NOT     NULL        ,  -- Environment required in server configuration (string validation as domain names)
 path         VARCHAR(256) NOT     NULL        ,  -- The full path to the directory (in development mode) or Jar (in production/development), or the base name if 'jar' BLOB is used
 jar          LONGBLOB     DEFAULT NULL        ,  -- Jar file of app, NULL if 'path' specifies the target directory or Jar file
 enabled      BOOLEAN      DEFAULT TRUE        ,  -- Flag for app enabled for use, FALSE if this app instance is disabled
 approval     SMALLINT     NOT NULL DEFAULT 1  ,  -- Approval state: 0=approved, 1=waiting (initial state), 2=processing, 3=rejected ('unapproved' text describes why app is rejected)
 unapproved   VARCHAR(100) DEFAULT NULL        ,  -- Text ID describing why app is not approved, non-NULL if rejected describing the problem.
 created      BIGINT       NOT     NULL        ,  -- Creation or last modification date of app entry or Jar, milliseconds since epoch in GMT, zero for system app
 UNIQUE (aid, version),
 FOREIGN KEY (aid) REFERENCES IZS.APPS(aid)  ON DELETE CASCADE
);


-- APP_GROUPS

-- App ID allow state for a group ID of users. By default, no Group ID is allowed access to any app and must explicitly be set.

CREATE TABLE IZS.APP_GROUPS (
 aid       BIGINT   NOT NULL,  -- App numeric ID
 gid       BIGINT   NOT NULL,  -- Group ID that is allowed access to this app 
 enabled   BOOLEAN  NOT NULL,  -- Group allowed flag.
 UNIQUE (aid, gid),
 FOREIGN KEY (aid) REFERENCES IZS.APPS  (aid) ON DELETE CASCADE,
 FOREIGN KEY (gid) REFERENCES IZS.GROUPS(gid) ON DELETE CASCADE
);


-- APP_USER_ENROLLED

-- App ID enrolled state for a user.

CREATE TABLE IZS.APP_USER_ENROLLED (
 aid       BIGINT    NOT NULL,            -- App numeric ID
 uid       BIGINT    NOT NULL,            -- User ID that is allowed access to this app
 enrolled  SMALLINT  NOT NULL DEFAULT 0,  -- Enrolled state: 0=requested, 1=processing, 2=auto-approved, 3=approved, 4=rejected
 enroll_ts BIGINT    NOT NULL,            -- The enrolled state last changed, milliseconds since epoch in GMT, zero for never
 UNIQUE (aid, uid),
 FOREIGN KEY (aid) REFERENCES IZS.APPS (aid) ON DELETE CASCADE,
 FOREIGN KEY (uid) REFERENCES IZS.USERS(uid) ON DELETE CASCADE
);


-- Insert default apps

-- TODO: INSERT INTO IZS.APPS (app_id, version, environment, path) VALUES


--------------------------------------------------------------------------------------------------------------


-- AGREEMENTS

-- An agreement text in a language, where the 'name' and 'language code' tuple must be unique.

CREATE TABLE IZS.AGREEMENTS (
 id          BIGINT         NOT     NULL PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
 name        VARCHAR(80)    NOT     NULL,  -- Name is in lower case, etc: "go", "srv", run"...
 lang_code   VARCHAR(100)   NOT     NULL,  -- BCP47: two letters, dash possible, etc, lower case... (can become long with variants)
 created     BIGINT         NOT     NULL,  -- Creation date since the epoch, 0 for none.
 descr       VARCHAR(300)   NOT     NULL,  -- Short description
 data        LONGBLOB       NOT     NULL,  -- Agreement text encoded in UTF-8
 is_html     BOOLEAN        NOT     NULL,  -- Flag indicating the agreement text is in HTML, FALSE means plain text
 updated_by  VARCHAR(80)    DEFAULT NULL,  -- Updated by another agreement 'name', NULL if not yet replaced
 UNIQUE (name, lang_code)                  -- Only one language code allowed per agreement
);


--------------------------------------------------------------------------------------------------------------


-- ACCEPTED AGREEMENTS

-- A user has accepted an agreement in a language.

CREATE TABLE IZS.ACC_AGR (
 id          BIGINT       NOT NULL,  -- The agreement accepted
 uid         BIGINT       NOT NULL,  -- User ID
 accepted    BIGINT       NOT NULL,  -- Date in epoch (GMT) milliseconds when user ID accepted
 ip_address  VARCHAR(41)  NOT NULL,  -- Max: IPv6 address [1234:5678:0123:4567:1234:5678:0123:4567]
 longitude   DOUBLE               ,  -- If non-NULL, the longitude of the IP address or the user's location when accepted
 latitude    DOUBLE               ,  -- If non-NULL, the latitude of the IP address or the user's location when accepted
 UNIQUE (id, uid)                 ,  -- User can only accept agreement once
 FOREIGN KEY (id)  REFERENCES IZS.AGREEMENTS(id) ON DELETE RESTRICT,
 FOREIGN KEY (uid) REFERENCES IZS.USERS(uid)     ON DELETE CASCADE
);


--------------------------------------------------------------------------------------------------------------


-- USER_FILES (max file size 2 GB), deleted when user is deleted.

CREATE TABLE IZS.USER_FILES (
 uid        BIGINT        NOT     NULL,  -- User ID
 created    BIGINT        NOT     NULL,  -- Date created in epoch milliseconds
 filename   VARCHAR(256)  NOT     NULL,  -- File name without path
 extension  VARCHAR(16)   NOT     NULL,  -- File extension
 size       INT           NOT     NULL,  -- File size
 crc32      INT           NOT     NULL,  -- File CRC32
 descr      VARCHAR(256)  DEFAULT NULL,  -- Description
 data       LONGBLOB      NOT     NULL,  -- The file data
 UNIQUE (uid, filename),
 FOREIGN KEY (uid) REFERENCES IZS.USERS(uid) ON DELETE CASCADE
);


-- GROUP_FILES (max file size 2 GB), deleted when group is deleted.

CREATE TABLE IZS.GROUP_FILES (
 gid        BIGINT        NOT     NULL,  -- Group ID
 created    BIGINT        NOT     NULL,  -- Date created in epoch milliseconds
 filename   VARCHAR(256)  NOT     NULL,  -- File name without path
 extension  VARCHAR(16)   NOT     NULL,  -- File extension
 size       INT           NOT     NULL,  -- File size
 crc32      INT           NOT     NULL,  -- File CRC32
 descr      VARCHAR(256)  DEFAULT NULL,  -- Description
 data       LONGBLOB      NOT     NULL,  -- The file data
 UNIQUE (gid, filename),
 FOREIGN KEY (gid) REFERENCES IZS.GROUPS(gid) ON DELETE CASCADE
);


-- FILES (common files: max file size 2 GB), manual deletion.

CREATE TABLE IZS.FILES (
 created    BIGINT        NOT     NULL,  -- Date created in epoch milliseconds
 filename   VARCHAR(256)  NOT     NULL,  -- File name without path
 extension  VARCHAR(16)   NOT     NULL,  -- File extension
 size       INT           NOT     NULL,  -- File size
 crc32      INT           NOT     NULL,  -- File CRC32
 descr      VARCHAR(256)  DEFAULT NULL,  -- Description
 data       LONGBLOB      NOT     NULL,  -- The file data
 UNIQUE (filename)
);


-- APP_FILES (appID files: max file size 2 GB), deleted when the application is deleted.

CREATE TABLE IZS.APP_FILES (
 aid        BIGINT        NOT     NULL,  -- App numberic ID
 filename   VARCHAR(256)  NOT     NULL,  -- File name without path
 extension  VARCHAR(16)   NOT     NULL,  -- File extension
 created    BIGINT        NOT     NULL,  -- Date created in epoch milliseconds
 size       INT           NOT     NULL,  -- File size
 crc32      INT           NOT     NULL,  -- File CRC32
 descr      VARCHAR(256)  DEFAULT NULL,  -- Description
 data       LONGBLOB      NOT     NULL,  -- The file data
 UNIQUE (aid, filename),
 FOREIGN KEY (aid) REFERENCES IZS.APPS(aid) ON DELETE CASCADE
);


-- EXT_FILES (external files using typically 'https:' URL's to CDN or similar where 2GB file size is not large enough [videos]).

CREATE TABLE IZS.EXT_FILES (
 eid         BIGINT        NOT     NULL PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,  -- Unique ID of external file
 created     BIGINT        NOT     NULL,  -- Date created in epoch milliseconds
 can_access  BOOLEAN       DEFAULT NULL,  -- Can access value: NULL=both server and client, FALSE=server only, TRUE=client only (e.g. server cannot access that URL for security reasons)
 url         VARCHAR(512)  NOT     NULL,  -- External file URL accessible from server and/or client
 extension   VARCHAR(16)   NOT     NULL,  -- File extension
 size_hint   BIGINT        DEFAULT NULL,  -- Hint of file size (preferably correct size), zero for unknown
 descr       VARCHAR(256)  DEFAULT NULL,  -- Description
 UNIQUE (url, can_access)
);


--------------------------------------------------------------------------------------------------------------


-- SUBSCRIPTIONS

-- User push notification subscriptions.
-- Subscriptions for a 'user' restrics removal of the user if subscriptions are present.
-- Subscriptions for an 'app_id' restrics removal of the app if subscriptions are present.
-- These restricts enforces that unsubscription on the remote side must be done BEFORE a user or app is removed.

CREATE TABLE IZS.SUBSCRIPTIONS (
 uid          BIGINT        NOT NULL,  -- User ID
 aid          BIGINT                ,  -- The App ID or NULL for all (like App ID = '*')
 topic        VARCHAR(253)          ,  -- The topic, NULL for none (instead of empty string)
 domain       VARCHAR(253)  NOT NULL,  -- Server domain name (host name), 'localhost' or alike for development
 created      BIGINT        NOT NULL,  -- Creation date in epoch milliseconds
 device_uuid  CHAR(20)      NOT NULL,  -- Device UUID (version 4) that created the subscription
 provider     VARCHAR(16)   NOT NULL,  -- Subscription provider e.g. 'VAPID' for VAPID, 'FBC' Firebase Cordova, 'APPLE' for Safari on macOS or iOS (not using Cordova)
 sub_id       VARCHAR(800)  NOT NULL,  -- Subscription ID from the client, string for some providers, JSON for e.g. VAPID
 UNIQUE (uid, aid, topic, domain),
 FOREIGN KEY (uid) REFERENCES IZS.USERS(uid) ON DELETE RESTRICT,
 FOREIGN KEY (aid) REFERENCES IZS.APPS (aid) ON DELETE RESTRICT
);
