/**
 * (C) Copyright Mindus SARL, 2026.
 * All rights reserved.
 *
 * @fileoverview Network presence detection.
 *
 * @author Christopher Mindus
 */

/* jshint unused:true, undef:true */
/* globals document, window, navigator, izBusy, log,
   XMLHttpRequest, clearTimeout, setTimeout */

(function(window,document,navigator)
  {
  // Defines that JavaScript code should be executed in "strict mode".
  "use strict";
  
  ///
  /// --- Variables ---
  ///
  
  // Debug flag.
  var DEBUG=false,
  
  // No connection retry.
  RETRY=5000,

  // Online flag: -1=unknown, 0=off-line, 1=on-line.
  online=-1,
  
  // Flag for listeners added.
  hasListeners,
  
  // Network connection Cordova object, undefined for browser.
  connection,
  
  // The URL to check: it should NOT refer to a page, just protocol,
  // host name and optional port, e.g. "https://someserver:port".
  urlDest,
  
  // Check timer for URL when connection should "come back".
  timer,
  
  // Text to show for no connection, falsy if auto-show is off.
  text,
  
  // The DIV element shown for the no connection.
  div,
  
  // The previous lock state to restore, null or undefined for none.
  prevLock,
  
  // Offline callback.
  offlineListener,
  
  // Callbacks to inform when network status comes back.
  listeners=[],
  
  // The different states reported by Cordova.
  states={};
  
  ///
  /// --- Functions ---
  ///
  
  /**
   * Adds the listener for online/offline.
   */
  function addListeners()
    {
    if ( !hasListeners )
      {
      hasListeners=true;
      
      window.addEventListener('online',function()
        {
        if ( DEBUG )
          log('network.online-listener called');
        
        checkState(1);
        });
      
      window.addEventListener('offline',function()
        {
        if ( DEBUG )
          log('network.offline-listener called');
 
        checkState(0);
        });
      }
    }
  
  /**
   * Gets the connection type.
   */
  function getType()
    {
    return connection?
        states[connection.type] || 'Undefined connection':
        online<0? 'Unchecked connection':
        online? 'On-line connection':
                'No network connection';
    }
  
  /**
   * Verifies the physical connection to a URL.
   */
  function checkURL(url,ok,err)
    {
    if ( DEBUG )
      log('network.checkURL: URL '+url);
    
    var r=new XMLHttpRequest();
    r.open('HEAD',url+'?'+(new Date()).getTime());
    r.timeout=5000;
    r.contentType='application/json';
    r.onreadystatechange=function()
      {
      if ( r.readyState==4 )
        {
        if ( DEBUG )
          log('network.checkURL: URL '+url+', status = '+r.status);
 
        if ( r.status>=200 )
          {
          ok();
          return;
          }
        
        err();
        }
      };
      
    // Send request.
    r.send();
    }
  
  /**
   * Updates the network state.
   */
  function checkState(state)
    {
    if ( DEBUG )
      log('network.checkState: destination = '+urlDest);
        
    if ( urlDest )
      potentialOnline();
    else
    if ( state )
      setOnline();
    else
      setOffline();
    }
  
  /**
   * Stops the timer if any.
   */
  function stopTimer()
    {
    if ( timer )
      clearTimeout(timer);
    
    timer=0;
    }
  
  /**
   * Potentially online. Check for real with timer.
   */
  function potentialOnline()
    {
    stopTimer();
    if ( urlDest )
      {
      if ( DEBUG )
        log('network.potentialOnline: checking '+urlDest);
        
      checkURL(urlDest,setOnline,function()
        {
        // Failed: start timeout in RETRY milliseonds.
        if ( DEBUG )
          log('network.potentialOnline: failed, retry in '+RETRY+'ms');
        
        setOffline();
        timer=setTimeout(potentialOnline,RETRY);
        });
      }
    else
      {
      if ( DEBUG )
        log('network.potentialOnline: no destination set, assuming on-line');
      
      setOnline();
      }
    }
  
  /**
   * Called when the connection goes online.
   */
  function setOnline()
    {
    if ( DEBUG )
      log('network.setOnline (destination = '+urlDest+'): '+getType());
      
    // Stop URL checking timer and set offline flag.
    stopTimer();
    online=0;
    
    // Remove divider.
    if ( removeDiv() )
      {
      izBusy.restore(prevLock);
      prevLock=null;
      }

    // Call any callbacks.
    for ( var cb;; )
      if ( (cb=listeners.shift()) )
        cb(1);
      else
        break;
    }
  
  /**
   * Called when the connection is gone offline.
   */
  function setOffline()
    {
    // Stop URL checking timer and set offline flag.
    stopTimer();
    online=0;
    
    // Automatic show of no connection?
    if ( text && !div )
      {
      div=document.createElement('DIV');
      div.className='izNC_offline';
      div.innerHTML='<div>'+text.htmlize()+'</div>';
      
      // Save previous lock state.
      prevLock=izBusy.save();
      
      // isBusy[,msg,[progress[,delay]]] [,onCancel])
      izBusy.setLock(true,null,-1,9e9);
      
      document.body.appendChild(div);
      setTimeout(function()
        {
        if ( div )
          div.className+=' izNC_show';
        },50);
      }
    
    // Listener?
    if ( offlineListener )
      offlineListener();
    }
  
  /**
   * Removes the div "slowly" but clears the "lock" immediately.
   */
  function removeDiv()
    {
    var d=div;
    div=null;
    if ( d )
      {
      // Make it slide down, after 350ms remove it.
      d.className='izNC_offline';
      setTimeout(function()
        {
        remove(d);
        },350);
      }
    
    return d;
    }
  
  /**
   * Removes an element.
   */
  function remove(element)
    {
    if ( element )
      {
      var p=element.parentNode,children=element.childNodes,ii;
      if ( p )
        p.removeChild(element);
  
      // Then remove it's children. This is done this way in order to avoid potential screen updates or layouts
      // requiring heavy CPU processing.
      if ( children && (ii=children.length) )
        while ( ii-- )
          remove(children[ii]);
      }
    }

  ///
  /// --- window.izNC singleton ---
  ///
  
  /**
   * The izNC (IIZI Network Connection) singleton.
   */
  window.izNC=
    {
    /**
     * Function init()
     * 
     * <p>Initializes the online/offline listeners for change of network state
     * used with iiziRun and Cordova (perhaps). 
     */
    init: function()
      {
      connection=navigator.connection;
      
      var c=window.Connection;
      if ( c )
        {
        states[c.UNKNOWN ]='Unknown connection';
        states[c.ETHERNET]='Ethernet connection';
        states[c.WIFI    ]='WiFi connection';
        states[c.CELL_2G ]='Cell 2G connection';
        states[c.CELL_3G ]='Cell 3G connection';
        states[c.CELL_4G ]='Cell 4G connection';
        states[c.CELL    ]='Cell generic connection';
        states[c.NONE    ]='No network connection';
        }
      
      // Add the listeners.
      addListeners();
      },
 
    /**
     * Function dest(url)
     * 
     * <p>Sets the destination for network state checking.
     * 
     * @type Function
     * 
     * @param {String} url  URL to check (only protocol+host+port)
     *                      as e.g. "https://server:port". Set to null to
     *                      stop checking a potential online situation.
     */
    dest: function(url)
      {
      // Set the destination URL and perform the check.
      urlDest=url;
      if ( !url )
        stopTimer();
      else // Add the listeners.
        addListeners();
      },
 
    /**
     * Function setAutoShow(ncText)
     * 
     * <p>Turns on or off showing of the network state.
     * 
     * @type Function
     * 
     * @param {String} ncText  If falsy, turns off automatic state display,
     *                         otherwise this should be the text to display in the
     *                         "toaster" when network connection is not present.
     */
    autoShow: function(ncText)
      {
      text=ncText;

      // No more text: remove off-line div and restore potential lock.
      if ( !ncText && removeDiv() )
        {
        izBusy.restore(prevLock);
        prevLock=null;
        }
      },
 
    /**
     * Function isOK(cb,recheck)
     * 
     * <p>In order for this function to work, the URL "dest(url)" function must have
     * been called. If not called, the callbacks will not be invoked and the
     * current online state is returned (1=online, 0=offline, -1=unknown).
     * 
     * <p>If "recheck" is false, the callbacks are not called. 
     * 
     * <p>Checks if the network connection is OK, and calls the callback if OK.
     * If not OK, it will periodically check the URL until OK.
     * 
     * @type Function
     * 
     * @param {Function}  cb         Callback function when connection has been verified.
     * @param {Boolean}   recheck    Flag to re-check even if on/offline is know.
     * @param {Function}  cbOffline  Sets or clears the callback that will be called upon offline, null for none.
     * 
     * @return {Number}  1 if network state is OK,
     *                   0 if not connected. If not known,
     *                  -1 if unknown and a potential destination URL will be checked.
     *                     If a destination URL is not defined, no callbacks will be invoked.
     */
    isOK: function(cb,recheck,cbOffline)
      {
      offlineListener=cbOffline;
      if ( !urlDest )
        return online;

      if ( !recheck && online>=0 )
        return online;
      
      listeners.push(cb);
      potentialOnline();
      return -1;
      },
      
    /**
     * Function getState()
     * 
     * <p>Gets the current online state.
     * 
     * @type Function
     * 
     * @return {Number}  1=online, 0=offline, -1=unknown.
     */
    getState: function()
      {
      return online;
      },
      
    /**
     * Function setOnline()
     * 
     * <p>Sets the state to online and removes a potential network DIV displaying no connection.
     * 
     * @type Function
     */
    setOnline: function()
      {
      setOnline();
      },
      
    /**
     * Function setOffline()
     * 
     * <p>Sets the state to offline and shows the potential network DIV displaying no connection.
     * 
     * @type Function
     */
    setOffline: function()
      {
      setOffline();
      }
    };
  
  // End of singleton, now start it...
  })(window,document,navigator);
