/**
 * (C) Copyright Mindus SARL, 2026.
 * All rights reserved.
 * 
 * The iiziGo FontInfo JavaScript counterpart.
 * 
 * @author Christopher Mindus
 */

/**
 * Function onClick()
 * 
 * @type Function
 *
 * Gets the font metrics and shows them.
 * 
 */
function onClick()
  {
  function get(id)
    {
    return document.getElementById(id).value;
    }
    
  function set(id,s)
    {
    return document.getElementById(id).innerHTML=s;
    }
  
  var family       =get('family'),
      size         =get('size'),
      weight       =get('weight'),
      style        =get('style'),
      lineHeight   =get('lineHeight'),
      letterSpacing=get('letterSpacing'),
      text         =document.getElementById('text').value,
      rc=getFontInfo(family,size,weight,style,lineHeight,letterSpacing),
      w=text? getStringWidth(family,size,weight,style,lineHeight,letterSpacing,text): -1;
  
  if ( w>0 )
    set('ostringWidth',w);

  set('oheight'        ,rc[0]);
  set('omaxWidth'      ,rc[1]);
  set('oaverageWidth'  ,rc[2]);
  set('oem'            ,rc[3]);
  set('oex'            ,rc[4]);
  set('opt'            ,rc[5]);
  set('oinputCharWidth',rc[6]);
  set('olineHeight'    ,rc[7]);
  }

/**
 * Function getStringWidth(text,family,size,weight,style,lineHeight,letterSpacing)
 * 
 * @param {String} text           HTML text to measure the width of.
 * @param {String} family         Font family, null for default.
 * @param {String} size           Font size with unit as in CSS, e.g. 12px, 110% or 14pt, null for default.
 * @param {String} weight         Font weight as in CSS, e.g. bold, thin or 800, null for default.
 * @param {String} style          Font style as in CSS, e.g. italic or oblique, null for default.
 * @param {String} lineHeight     Line height as in CSS, null for default.
 * @param {String} letterSpacing  Letter spacing as in CSS, null for default.
 * 
 * @return {Number} Pixel width.
 */
function getStringWidth(family,size,weight,style,lineHeight,letterSpacing,text)
  {
  var t=document.getElementById('measure'),
      s=t.style;
  
  s.fontFamily   =family        ||'';
  s.fontSize     =size          ||'';
  s.fontWeight   =weight        ||'';
  s.fontStyle    =style         ||'';
  s.lineHeight   =lineHeight    ||'';
  s.letterSpacing=letterSpacing ||'';
  t.innerHTML    =text          ||'';
  
  return t.offsetWidth;
  }

/**
 * The point size factor for the PT unit to pixels.
 */
var pointFactor;

/**
 * Function getPointFactor()
 * 
 * @type Function
 * 
 * Calculates the point factor.
 * 
 * @return {Number}   The point factor.
 */
function getPointFactor()
  {
  if ( !pointFactor )
    {
    var d=izCreateElement(DIV),
        s=d.style;
    
    s.position=ABSOLUTE;
    s.top=s.left='-9999px';
    s.width=s.height='1in';
    document.body.appendChild(d);
    pointFactor=d.offsetHeight/72; // Factor is 1/72 inch.
    izDestroyElement(d);
    }
  
  return pointFactor;
  }

/**
 * Function getFontInfo(family,size,weight,style,lineHeight,letterSpacing)
 * 
 * @type Function
 * 
 * @param {String} family         Font family, null for default.
 * @param {String} size           Font size with unit as in CSS, e.g. 12px, 110% or 14pt, null for default.
 * @param {String} weight         Font weight as in CSS, e.g. bold, thin or 800, null for default.
 * @param {String} style          Font style as in CSS, e.g. italic or oblique, null for default.
 * @param {String} lineHeight     Line height as in CSS, null for default.
 * @param {String} letterSpacing  Letter spacing as in CSS, null for default.
 * 
 * @return {Array} of Number's:
 *  <ul>
 *  <li>[0] = height        
 *  <li>[1] = maxWidth  
 *  <li>[2] = averageWidth  
 *  <li>[3] = em            
 *  <li>[4] = ex            
 *  <li>[5] = pt            
 *  <li>[6] = inputCharWidth
 *  <li>[7] = lineHeight
 *  </ul>
 */
function getFontInfo(family,size,weight,style,lineHeight,letterSpacing)
  {
  var f=new FontMetrics(family        || '',
                        size          || '',
                        weight        || '',
                        style         || '',
                        lineHeight    || '',
                        letterSpacing || '',
                        true); // doCalcLineHeight!
  
  return [f.height,                  // 0
          f.maxWidth,                // 1
          f.avgWidth,                // 2
          f.emWidth,                 // 3
          f.exHeight,                // 4
          getPointFactor(),          // 5
          (f.avgWidth+f.emWidth)/2,  // 6
          f.lineHeight               // 7
          ];
  }

// Asynchronous events are only fired if the browser is ready.
// The browser is not ready when loading.
setTimeout(function()
  {
  // Callback to Eclipse when done.
  try { onJSInitialized(); }
  catch(e) {}
  },0);
