/**
 * (C) Copyright Mindus SARL, 2026.
 * All rights reserved.
 *
 * @fileoverview Busy state functions with message and progress support.
 *
 * @author Christopher Mindus
 */

/* jshint unused:true, undef:true */
/* globals document, window */

(function(window,document)
  {
  // Defines that JavaScript code should be executed in "strict mode".
  "use strict";
  
  // Creates an element.
  function createElement(type)
    {
    return document.createElement(type);
    }
    
  // Creates a DIV.
  function createDiv()
    {
    return createElement('div');
    }
    
  // Gets an element by ID.
  function byId(id)
    {
    return document.getElementById(id);
    }

  // Removes an element.
  function remove(element)
    {
    if ( element )
      {
      var p=element.parentNode,children=element.childNodes,ii;
      if ( p )
        p.removeChild(element);
  
      // Then remove it's children. This is done this way in order to avoid potential screen updates or layouts
      // requiring heavy CPU processing.
      if ( children && (ii=children.length) )
        while ( ii-- )
          remove(children[ii]);
      }
    }
  
  // Current busy state.
  var busy=false,
  
  // The message, null for none.
  message=null,
  
  // The current progress, -1 for none.
  prog=-1,
  
  // The cancel callback registered, null for none.
  cbCancel=null,
  
  // The lock DIV at the first position in document.
  div,div2,div3,div4,
  
  // The message div.
  mDiv,
  
  // The progress bar div.
  pDiv,pDiv2,
  
  // The last arguments saved, null for no lock.
  saveArgs=null, // Comma!
 
  /**
   * Property window.izBusy
   * 
   * <p>The IIZI Busy functions object.
   * 
   * @type Object
   */
  izBusy=window.izBusy=
    {
    /**
     * Function setLock(isBusy[,msg,[progress[,delay]]] [,onCancel])
     * 
     * <p>Sets or clears the busy state. The busy state may be set with a message and optional progress indicator.
     * The progress indicator value is a value between 0-100.
     * 
     * @type Function
     * 
     * @param {Boolean}         isBusy          The busy state.
     * @param {String}          msg             Optional message to display as a HTML string, escape to show any message.
     * @param {Number}          progress        Optional progress value between 0 and 100 (no progress for other values is assumed).
     * @param {Function|Number} onCancel|delay  Function: Optional cancel function that is called if mouse or touch is pressed.
     *                                          Number: Delay in milliseconds before wait is shown, default 1.1 second.
     */
    setLock: function(isBusy,msg,progress,option)
      {
      // Set message or null.
      var v=!!isBusy,b,s,
          delay   =v && typeof option=='number'  ? option: -2,
          cbCancel=v && typeof option=='function'? option: null;

      // Save the last arguments: null for no lock.
      saveArgs=v? Array.prototype.slice.call(arguments): null;
      message=v && msg? msg: null;
      prog=v? progress: -1;
      if ( v!=busy )
        {
        // Change of state.
        busy=v;
        if ( v )
          {
          if ( !div )
            {
            // Locked: create busy.
            div=createDiv();
            div.className='izr_lock izr_locked';
            div.tabIndex=1e7;
            div .appendChild(div2=createDiv());
            div2.appendChild(div3=createDiv());
            div3.appendChild(div4=createDiv());
            
            // Cancel function (store function for retrieval)?
            if ( cbCancel )
              {
              div.onclick=function()
                {
                if ( cbCancel && busy )
                  {
                  izBusy.setLock();
                  cbCancel();
                  }
                };
              }
              
            // Add spinner.
            div4.appendChild(b=createDiv());
            b.className='izr_lock1';
            b.appendChild(v=createDiv());
            v.className='izr_lock2';
            for ( b=8; b--; )
              v.appendChild(createDiv());
              
            // Add first in body.
            b=document.body;
            v=b.firstElementChild;
            if ( v )
              b.insertBefore(div,v);
            else
              b.appendChild(div);
            
            // Focus the div.
            div.focus();
            }
          }
        else
        if ( div )
          {
          // Remove busy state immediately along with message, progress and callback.
          div2=div3=div4=mDiv=pDiv=pDiv2=cbCancel=div.onclick=null;
          remove(div);
          div=0;
          }
        }
   
      // If busy, also process optional message and progress bar.
      if ( busy )
        {
        // Delay?
        s=byId('izr_busy_css');
        if ( delay>-2 )
          {
          if ( !s )
            {
            s=createElement('style');
            s.type='text/css';
            s.id='izr_busy_css';
            document.getElementsByTagName('head')[0].appendChild(s);
            }
          
          // Assign delay.
          delay=delay<0? '999999s': delay+'ms';
          s.innerHTML='.izr_locked{-webkit-animation-delay:'+delay+
                                         ';animation-delay:'+delay+';}';
          }
        
        // Progress?
        if ( typeof progress=='number' && progress>=0 && progress<=100 )
          {
          if ( !pDiv )
            {
            // Create 2 div's.
            pDiv=createDiv();
            pDiv.className='izr_progress';
            pDiv.appendChild(pDiv2=createDiv());
            
            // Add before message, or last if no message.
            if ( mDiv )
              div4.insertBefore(pDiv,mDiv);
            else
              div4.appendChild(pDiv);
            }
          
          // Update the progress.
          pDiv2.style.width=progress+'%';
          }
        else // Remove progress?
        if ( pDiv )
          {
          remove(pDiv);
          pDiv=pDiv2=0;
          }
        
        // Message?
        if ( msg )
          {
          if ( !mDiv )
            {
            div4.appendChild(mDiv=createDiv());
            mDiv.className='izr_msg';
            }
          
          mDiv.innerHTML=msg;
          }
        else
        if ( mDiv )
          {
          // Remove previous message?
          remove(mDiv);
          mDiv=0;
          }
        }
      else
        {
        // Remove styles for delay upon clear of busy.
        if ( (s=byId('izr_busy_css')) )
          remove(s);
        }
      },
      
    /**
     * Function isLocked()
     * 
     * @type Function
     * 
     * @return {Boolean} The busy flag.
     */
    isLocked: function()
      {
      return busy;
      },
      
    /**
     * Function getMessage()
     * 
     * @type Function
     * 
     * @return {String} The current message.
     */
    getMessage: function()
      {
      return message;
      },
      
    /**
     * Function getProgress()
     * 
     * @type Function
     * 
     * @return {Number} The progress value, -1 for none.
     */
    getProgress: function()
      {
      return prog;
      },
      
    /**
     * Function getCancel()
     * 
     * <p>Gets the cancel function that was set when lock was first put in place.
     * 
     *  @type function
     *  
     *  @return {Function} The cancel function, or null for none.
     */
    getCancel: function()
      {
      return cbCancel;
      },
      
    /**
     * Function save(clear)
     * 
     * <p>Saves all the settings so a lock state can be restored.
     * 
     * @type Function
     * 
     * @param {Boolean}  Optional flag to clear the lock state if true'ish.
     * 
     * @return {Array}  The arguments to last call to "setLock(...)", null for no lock state.
     */
    save: function(clear)
      {
      var s=saveArgs;
      if ( clear )
        izBusy.setLock();
      
      return s;
      },
      
    /**
     * Function restore(s)
     * 
     * <p>Restores a saved lock state.
     * 
     * @type Function
     * 
     * @param {Object} s  The saved lock state, null for no lock.
     */
    restore: function(s)
      {
      if ( s )
        izBusy.setLock.apply(izBusy,s);
      else
        izBusy.setLock();
      }
    };
  })(window,document);