#!/usr/bin/env node

/* jshint esversion:6, latedef:nofunc, node:true */

// Exit if not ULGIFY is set.
const isRelease = process.env.UGLIFY == 'true';
if ( !isRelease )
  {
  console.log('\n-----\nCompressing of files ignored: NOT RELEASE VERSION:\n'+JSON.stringify(process.env,null,2)+'\n'+'\n-----\n');
  return;
  }
  
// Modules
const readlineSync = require('readline-sync');
const fs = require('fs');
const path = require('path');
const cwd = process.cwd();
const dependencyPath = path.join(cwd, 'node_modules');

// Dependencies.
const UglifyJS = require(path.join(dependencyPath, 'uglify-js'));
const CleanCSS = require(path.join(dependencyPath, 'clean-css'));

// Process.
const platform = process.env.PLATFORM;
const platformPath = path.join(cwd, 'platforms');

// Hook configuration.
const configFilePath = path.join(cwd, 'scripts', 'compress-settings.json');
const hookConfig = JSON.parse(fs.readFileSync(configFilePath));
const foldersToProcess = hookConfig.foldersToProcess; // add other www folders in here if needed (ex. js/controllers)
const uglifyJsOptions = hookConfig["uglify-js-options"];
const cssMinifier = new CleanCSS(hookConfig["clean-css-options"]);

// Get the option for minifying HTML.
const htmlMinify = require('html-minifier').minify;
const htmlOptions = hookConfig["html-minifier-options"];

// SVGO.
const { optimize } = require('svgo');

// Get file paths endings to ignore.
const ignoreEndingFilePaths = hookConfig.ignoreEndingFilePaths;
const ignoreEndingFilePathsWarnings = hookConfig.ignoreEndingFilePathsWarnings;

// Run uglifier
var css=0,js=0,html=0,svg=0;

try
  {
  console.log('\n-----\n\nCompressing files (version 1.0.1):\n');
  run();
  console.log('\nSUCCESS: compressed '+html+' HTML, '+css+' CSS, '+svg+' SVG, and '+js+' JS files\n-----\n');
  return 0;
  }
catch(e)
  {
  console.error('\n\n***** FAILED Compressing *****\n\n',e);
  throw e;  
  }

/**
 * Run compression for all specified platform.
 */
function run()
  {
  var wwwPath;
  switch(platform)
    {
    case 'android':
      wwwPath = path.join(platformPath, platform, 'app', 'src', 'main', 'assets', 'www');
      break;

    case 'ios':
      wwwPath = path.join(platformPath, platform, 'www');
      break;

    default:
      console.log('This hook only supports "android" and "ios": platform = "'+platform+'"');
      throw new Error('Unsupported platform "'+platform+'"');
    }

  if ( !processFolders(wwwPath) )
    throw new Error('Failed processing folder "'+wwwPath+'"');
  }

/**
 * Processes defined folders.
 * 
 * @param  {string} wwwPath - Path to www directory
 * 
 * @return {boolean} Success flag.
 */
function processFolders(wwwPath)
  {
  var ok=true;
  foldersToProcess.forEach(function(folder)
    {
    if ( ok )
      {
      if ( folder==='.' )
        ok=processFiles(wwwPath, false);
      else
        ok=processFiles(path.join(wwwPath, folder), true);
      }
    });
  
  return ok;
  }

/**
 * Processes files in directories.
 * 
 * @param  {string}  dir      Directory path
 * @param  {boolean} recurse  Flag to recurse or not in directories
 * 
 * @return {boolean} Success flag.
 */
function processFiles(dir, recurse)
  {
  if ( !fs.existsSync(dir) )
    {
    console.log(' - DIR : ' + dir + ' --> ignored, not found on this platform');
    return true;
    }
 
  var ok=true;
  fs.readdirSync(dir).forEach(function(file)
    {
    if ( ok )
      {
      file = path.join(dir, file);
      var stat=fs.statSync(file);
      if ( stat.isFile() )
        compress(file);
      else
      if ( recurse && stat.isDirectory() )
        ok=processFiles(file, true);
      }
    });
  
  return ok;
  }

/**
 * Compresses file.
 * 
 * @param  {string} file - File path
 * 
 * @return {undefined}
 */
function compress(file)
  {
  var ext = path.extname(file),
      source, result, empty,
      fnNative=file.toString(),
      fn=fnNative.replace(/\\/g,'/'), ii;
  
  if ( ignoreEndingFilePaths )
    for ( ii=ignoreEndingFilePaths.length; ii--; )
      if ( fn.endsWith(ignoreEndingFilePaths[ii]) )
        {
        console.log(' - SKIP: '+fn);
        return;
        }
  
  function ignoreWarnings()
    {
    if ( ignoreEndingFilePathsWarnings )
      for ( ii=ignoreEndingFilePathsWarnings.length; ii--; )
        if ( fn.endsWith(ignoreEndingFilePathsWarnings[ii]) )
          {
          console.log('   --- Ignoring warnings ---');
          return true;
          }

    return false;
    }

  switch(ext)
    {
    case '.js':
      console.log(' - JS  : ' + fn);
      source = fs.readFileSync(file, 'utf8').trim();
      empty = (!source);
      if ( empty )
        result='';
      else
        {
        result = UglifyJS.minify(source, uglifyJsOptions);
        if ( result.error )
          {
          console.log('   ***** ERROR UglifyJS("'+fn+'"): '+result.error);
          result='';
          }
        else
          {
          if ( !ignoreWarnings() )
            {
            var w=result.warnings,cc,j;
            if ( w && (cc=w.length) )
              {
              console.log('     *** WARNINGS ***');
              for ( j=0; j<cc; ++j )
                console.log('      > '+w[j]);
              }
            }
          
          result=(result.code)? result.code.trim(): '';
          }
        }
      ++js;
      break;

    case '.css':
      console.log(' - CSS : ' + fn);
      source = fs.readFileSync(file, 'utf8').trim();
      empty = (!source);
      if ( empty )
        result='';
      else
        {
        var parse={};
        parse[fnNative]={styles: source};
        result = cssMinifier.minify(parse);
        var rc='['+(''+result.errors).trim()+']';
        if ( rc!='[undefined]' && rc!='[]' )
          {
          result='';
          console.log('   ***** ERROR cssMinifier("'+fn+'"): '+rc);
          }
        
        result=(result.styles)? result.styles.trim(): '';
        }
      ++css;
      break;

    case '.htm':
    case '.html':
      console.log(' - HTML: ' + fn);
      source = fs.readFileSync(file, 'utf8').trim();
      empty = (!source);
      result = (empty)? '': htmlMinify(source, htmlOptions).trim();
      ++html;
      break;

    case '.svg':
      if ( fn.endsWith('fontawesome-webfont.svg') )
        {
        console.log(' - SKIP: ' + fn);
        return;
        }
        
      try
        {
        console.log(' - SVG : ' + fn);
        source = fs.readFileSync(file, 'utf8').trim();
        empty = (!source);
        result = (empty)? '': optimize(source, { path: fn, multipass: true }).data;
        ++svg;
        }
      catch(e)
        {
        console.error('FAILED SVGO compression, fn = '+fn,e);
        }
      break;

    default:
      // console.log(indent + ' - Skip (' + ext + '): '+file);
      return;
    }
  
  // Check if empty result: it is an error!
  if ( !empty && !result && !ignoreWarnings() &&
    !readlineSync.keyInYN('\n   ***** ERROR: No result to write in "'+fn+'"\n\nDo you wish to proceed?') )
    {
    console.log('\n\n*** ABORTING ***\n\n');
    throw new Error('Compression aborted by user');
    }

  // Overwrite the original unminified file.
  fs.writeFileSync(file, result, 'utf8');
}