/**
 * (C) Copyright Mindus SARL, 2025.
 * All rights reserved.
 * 
 * Routines to provide pop-up dialog box support regardless of Dojo and Themes.
 * 
 * @author Christopher Mindus
 */

/* jshint unused:true, undef:true */
/* globals window, document, setTimeout */

(function(window,document)
  {
  var cover,
      popup,
      domNode,
      blur=1,
      selectedItem,
      cbBlock,
 
      // Common strings.
      mouseup    ='mouseup',
      mousedown  ='mousedown',
      mouseout   ='mouseout',
      touchstart ='touchstart',
      touchend   ='touchend',
      touchcancel='touchcancel',
      touchmove  ='touchmove',
      NONE       ='none',
      DIV        ='div',
      INPUT      ='input',
      BUTTON     ='button',
      SPACE      =' ',
      ON         ='on',
      
      izrpBlur   =' izrpBlur',
      izrpUnblur =' izrpUnblur',
          
      // Events.
      events =[mousedown,'mousemove',mouseup,'mouseover',mouseout,'DOMMouseScroll',
               touchstart,'touchforcechange',touchmove,touchend,touchcancel],
      release=[mouseup,mouseout,touchend,touchcancel],
      press  =[mousedown,touchstart];
  
  /**
   * Function addClass(element)
   * 
   * <p>Adds a class to an element.
   * 
   * @type Function
   * 
   * @param {HTMLElement} element  The element.
   * @param {String}      cn       Class.
   */
  function addClass(element,cn)
    {
    var n=element.className.split(SPACE);
    if ( n.indexOf(cn)<0 )
      {
      n.push(cn);
      element.className=n.join(SPACE);
      }
    }

  /**
   * Function removeClass(element)
   * 
   * <p>Removes a class from an element.
   * 
   * @type Function
   * 
   * @param {HTMLElement} element  The element.
   * @param {String}      cn       Class.
   */
  function removeClass(element,cn)
    {
    var n=element.className.split(SPACE),
        ii=n.indexOf(cn);
    
    if ( ii>=0 )
      {
      n.splice(ii,1);
      element.className=n.join(SPACE);
      }
    }

  /**
   * Function destroyElement(element)
   * 
   * @type Function
   * 
   * @param {HTMLElement} element  The element to destroy.
   * 
   * <p>Function to "safely" destroy an HTMLElement that has been allocated. There are many memory
   * leaks in various browsers, so it's best to have it all located in one area.
   * 
   * <p>Additional memory leak protection code could be added here as well to destroy all on* properties
   * of the node and detach all of the node's event listeners.
   */
  function destroyElement(element)
    {
    // Remove element.
    if ( element )
      {
      var p=element.parentNode,
          children=element.childNodes,ii;
      
      if ( p )
        p.removeChild(element);
  
      // Then remove it's children. This is done this way in order to avoid potential screen updates or layouts
      // requiring heavy CPU processing.
      if ( children && (ii=children.length) )
        while ( ii-- )
          destroyElement(children[ii]);
      }
    }
    
  /**
   * Function setSelectable(node,selectable)
   * 
   * <p>Sets the given node as selectable or unselectable.
   * 
   * @type Function
   * 
   * @param {HTMLElement} node        The node.
   * @param {Boolean}     selectable  The selectable flag.
   */
  function setSelectable(node,selectable)
    {
    if ( node )
      {
      node.style.KhtmlUserSelect=selectable? 'auto': NONE;
      node.style.MozUserSelect  =selectable? '': NONE;
      node.onselectstart=selectable? null: function() { return false; };
      if ( node.unselectable!==undefined )
        {
        // For IE.
        node.unselectable=selectable? '': ON;
        for ( var nodes=node.getElementsByTagName('*'), ii=nodes.length; ii--; )
          nodes[ii].unselectable=selectable? '': ON;
        }
      }
    }
  
  /**
   * Creates an element with the specified class name.
   * 
   * @type Function
   * 
   * @param {String} type  Element type to create.
   * @param {String} cls   Class name to set to element.
   */
  function createElement(type,cls)
    {
    var node=document.createElement(type);
    node.className=cls;
    return node;
    }
  
  /**
   * Function cancelSelection()
   * 
   * <p>Cancels the selected item.
   * 
   * @type Function
   */
  function cancelSelection()
    {
    if ( selectedItem )
      {
      removeClass(selectedItem,'izrpSel');
      selectedItem=0;
      }
    }
  
  /**
   * Function blocker(e)
   * 
   * <p>The blocker function for cover.
   * 
   * @type Function
   * 
   * @param {Event}  e  The mouse or touch event.
   */
  function blocker(e)
    {
    if ( !cbBlock || !cbBlock(e) )
      {
      if ( e.cancelable )
        e.preventDefault();
 
      return (e.returnCode=false);
      }
    }
  
  /// ------------------------
  /// --- Exposed routines ---
  /// ------------------------

  /**
   * Property izrPrompt
   * 
   * <p>The Object for the iiziRun Prompt routines.
   * 
   * @type Object
   */
  window.izrPrompt=
    {
    /**
     * Function blur(on)
     * 
     * <p>Sets the blur flag to use for the next prompt.
     * 
     * @type Function
     * 
     * @param {Boolean} on  The blur flag.
     */
    blur: function(on)
      {
      blur=on;
      },

    /**
     * Function register(f)
     * 
     * <p>Registers all mouse and touch events to handle the cover.
     * 
     * @type Function
     * 
     * @param {Function} f  The function to call, null to unregister.
     *                      If the function returns true, no further processing is done
     *                      to prevent defaults, etc. Returning false will stop the default
     *                      event processing and return false.
     *                      The function parameter is the event.
     * @param {Boolean} ns  Cancel doument body scrolling.
     */
    register: function(f/*,ns*/)
      {
      var ii=events.length;//,b=document.body;
      cbBlock=f;
      if ( f )
        {
        while ( ii-- )
          document.addEventListener(events[ii],blocker);
        }
      else
        while ( ii-- )
          document.removeEventListener(events[ii],blocker);
      },
      
    /**
     * isPressEvent(e)
     * 
     * <p>Checks if an event is of a press-type for the blocker event.
     * 
     * @type Function
     * 
     * @param {Event} e  The mouse or touch event.
     */
    isPressEvent: function(e)
      {
      return press.indexOf(e.type)>=0;
      },

    /**
     * isReleaseEvent(e)
     * 
     * <p>Checks if an event is of a release-type for the blocker event
     * (mouse up or out, touch end or cancel).
     * 
     * @type Function
     * 
     * @param {Event} e  The mouse or touch event.
     */
    isReleaseEvent: function(e)
      {
      return release.indexOf(e.type)>=0;
      },

    /**
     * Function prompt(parent,title,prompt,fields,okText,cancelText,cbDone,cbCancel)
     * 
     * <p>Opens a pop-up dialog on the page. The <code>fields</object> array consists of the
     * follow Object:
     * 
     * <pre>
     *   {
     *   t: textPrompt,  // {String}
     *   h: hint,        // {String}
     *   s: size,        // {Number}
     *   m: maxLength,   // {Number} where 0=no limit
     *   p: password     // {Boolean}
     *   o: omit         // {Boolean} optional flag indicating value can be empty.
     *   v: value        // {String}  optional initial value.
     *   }
     * </pre>
     * 
     * @type Function
     * 
     * @param {HTMLElement} parent      Parent node to disable, null for none. 
     * @param {String}      title       The title string in HTML.
     * @param {String}      prompt      The prompt of the top dialog part in HTML.
     * @param {Array}       fields      The fields array Objects.
     * @param {String}      okText      Text for OK button, null if cancelText is an array of buttons.
     * @param {String}      cancelText  Text for Cancel button, or an array of buttons if okText is null.
     * @param {Function}    cbDone      Callback for "done". Returns in the fields[].v="text content" input variable.
     * @param {Function}    cbCancel    Callback for "cancel".
     */
    prompt: function(parent,title,prompt,fields,okText,cancelText,cbDone,cbCancel)
      {
      // Dialog already processing?
      if ( cover )
        {
        // Cancel this dialog, not the other one.
        cbCancel();
        return;
        }
      
      // Save dom node for parent, can be null.
      domNode=parent;
      
      var self=this,
          t=createElement('table','izrpTable'),
          r=t.insertRow(),
          c=r.insertCell(),
          ii=0,d,e,f,ok,focus,head;
      
      c.className='izrpTitle';
      c.colSpan=2;
      c.innerHTML=title;
      
      r=t.insertRow();
      head=r.insertCell();
      d=prompt.indexOf('\t');
      if ( d<0 )
        {
        head.innerHTML=prompt;
        head.colSpan=2;
        }
      else
        {
        // Icon and message.
        head.innerHTML=prompt.substr(0,d);
        head.className='izrpIcon';
        
        head=r.insertCell();
        head.innerHTML=prompt.substr(d+1);
        }
      
      head.className='izrpHead';
      
      // Check all values present.
      function check()
        {
        if ( cover )
          {
          for ( var jj=fields.length, on=true; jj--; )
            {
            f=fields[jj].e;
            if ( !f )
              return;
  
            // Can input be omitted?
            if ( !f.o )
              on&=!!f.value.length;
            }

          if ( ok )
            {
            ok.disabled=on? '': 'disabled';
            ok.style.opacity=on? 1: 0.5;
            }
          }
        }
      
      while ( ii<fields.length )
        {
        // Prompt.
        f=fields[ii++];
        r=t.insertRow();
        c=r.insertCell();
        c.className='izrpPrompt';
        c.innerHTML=f.t;
 
        // Entry.
        c=r.insertCell();
        c.className='izrpEntryC';
 
        f.e=e=createElement(INPUT,'izrpEntryF');
        e.type=f.p? 'password': 'text';
        e.placeholder=f.h;
        e.size=f.s;
        if ( e.m )
          e.maxLength=e.m;
        
        focus=focus || e;
        
        if ( f.v )
          e.setAttribute('value',f.v);
 
        c.appendChild(e);
        
        e.onkeyup=
        e.onpaste=
        e.onclick=
        e.onpropertychange=
        e.oninput=
        e.onchange=check;
        }

      r=t.insertRow();
      c=r.insertCell();
      c.colSpan=2;
      d=createElement(DIV,'izrpButtons');
      c.appendChild(d);

      // OK button.
      function doOK(e)
        {
        if ( cover )
          {
          // OK not enabled?
          if ( ok && ok.disabled )
            return;
          
          for ( ii=fields.length; ii--; )
            {
            f=fields[ii];
            f.v=f.e.value;
            f.e=null;
            }
 
          // Find button that was pressed. None? use default first!
          ii=0;
          try { ii=(e.target||e.srcElement)._ii; }
          catch(x) {}
          self.removeCover(function()
            {
            cbDone(ii);
            });
          }
        }
      
      // Cancel button.
      function doCancel()
        {
        // Cancel clicked.
        if ( cover )
          self.removeCover(cbCancel);
        }
      
      if ( okText )
        {
        // OK and cancel button.
        e=createElement(DIV,'izrpOK');
        d.appendChild(e);
        ok=createElement(INPUT,'');
        ok._ii=0;
        ok.disabled='disabled';
        ok.style.opacity=0.5;
        ok.type=BUTTON;
        ok.value=okText;
        ok.onclick=doOK;
        e.appendChild(ok);
  
        e=createElement(DIV,'izrpCancel');
        d.appendChild(e);
        f=createElement(INPUT,'');
        f.type=BUTTON;
        f.value=cancelText;
        f.onclick=doCancel;
        e.appendChild(f);
        }
      else
        {
        // Array of buttons, the first one is default
        head.className+=' izrpMsg';
        for ( var cn='izrpOK',n=0; n<cancelText.length; ++n )
          {
          e=createElement(DIV,cn);
          d.appendChild(e);
          f=createElement(INPUT,'');
          f._ii=n;
          f.type=BUTTON;
          f.value=cancelText[n];
          f.onclick=doOK;
          e.appendChild(f);
          
          focus=focus || f;
          
          cn='izrpCancel';
          }
        }
        
      // Create cover and dialog box.
      self.addCover();
      cover.appendChild(e=createElement(DIV,'izrpOuter'));
      e.appendChild(popup=createElement(DIV,'izrpInner'));
      popup.appendChild(t);
 
      // Focus first entry field.
      focus.focus();
      
      // Keys in dialog box.
      cover.onkeydown=function(e)
        {
        // Escape key in dialog box.
        e=e.keyCode;
        if ( e==27 )
          doCancel();
        else
        if ( e==13 )
          doOK(0);
        };
        
      // Check a little later in case there is auto-prompt in browser that
      // fills in data.
      setTimeout(check,400);
      },
      
    /**
     * Function list(parent,title,items,cbDone[,cbCancel])
     * 
     * <p>Shows a pop-up selection of scrollable list items to select.
     * 
     * @param {HTMLElement} parent      Parent node to disable, null for none. 
     * @param {String}      title       The title string in HTML, null for none.
     * @param {Array}       items       Strings or Object's that makes up the entries to select. If "Object",
     *                                  the member "t" is the text and "i" is the icon HTML. 
     * @param {Function}    cbDone      Callback for "done". Returns in the fields[].v="text content" input variable.
     * @param {Function}    cbCancel    Optional callback for "cancel", null for none.
     */
    list: function(parent,title,items,cbDone,cbCancel)
      {
      // Dialog already processing?
      if ( cover )
        {
        // Cancel this dialog, not the other one.
        cbCancel();
        return;
        }
      
      // Save dom node for parent, can be null.
      domNode=parent;
      
      var self=this,
          t=createElement('table','izrpTableList'),
          e,icon,cc=items.length,ii=cc,lastScroll;
      
      // Check if any icon is present.
      while ( ii-- )
        if ( items[ii].i )
          {
          icon=1;
          break;
          }

      // Create title if required.
      if ( title )
        {
        var r=t.insertRow(),c;
        c=r.insertCell();
        c.className='izrpTitleList';
        c.innerHTML=title;
        if ( icon )
          c.colSpan=2;
        }
      
      // Creates an item.
      function crt(item,index)
        {
        var row=t.insertRow(),c,x0,y0;
        row._i=ii;
        setSelectable(row,false);
        
        if ( index )
          row.className='izrpSep';
 
        if ( icon )
          {
          // Icon element.
          c=row.insertCell();
          c.className='izrpListIcon';
          c.innerHTML=item.i||'&nbsp;';
          setSelectable(c,false);
          }
        
        // Add text.
        c=row.insertCell();
        c.className='izrpListItem';
        c.innerHTML=item.t || item || '';
        setSelectable(c,false);
        
        // Processing of clicks.
        row.onclick=function(e)
          {
          if ( row==selectedItem )
            self.removeCover(function() { cbDone(index); });
          else
            cancelSelection();
 
          e.preventDefault();
          return false;
          };
         
        // TouchStart and MouseDown.
        row.ontouchstart=row.onmousedown=function(e)
          {
          x0=e.offsetX;
          y0=e.offsetY;
          cancelSelection();
          
          //console.log('start = '+x0+','+y0);
          
          //e.preventDefault();
          //e.stopPropagation();
          
          if ( !lastScroll || Date.now()>lastScroll+150 )
            addClass(selectedItem=row,'izrpSel');
          };
         
          
        // Move events.
        row.ontouchmove=row.onmousemove=function(e)
          {
          if ( selectedItem==row )
            {
            var x=e.offsetX,
                y=e.offsetY;
            
            //console.log('move = '+x+','+y);
            
            // Moved too far away (5 pixels), or outside?
            if ( Math.abs(x0-x)>5 || Math.abs(y0-y)>5 ||
                 x<0 || x>=row.offsetWidth ||
                 y<0 || y>=row.offsetHeight )
              cancelSelection();
            }
          };

        // Outside.
        row.onmouseout=row.ontouchcancel=function()
          {
          cancelSelection();
          };
          
        // Released.
        row.onmouseup=row.ontouchend=function()
          {
          // Inside?
          if ( selectedItem==row )
            self.removeCover(function() { cbDone(index); });
          else
            cancelSelection();
          };
        }
      
      // Create all items.
      for ( ii=0; ii<cc; ++ii )
        crt(items[ii],ii);
 
      // Create cover with cancel function.
      self.addCover(function()
        {
        self.removeCover(cbCancel);
        });
      
      // Add popup.
      cover.appendChild(e=createElement(DIV,'izrpOuter izrpCenter'));
      e.appendChild(popup=createElement(DIV,'izrpInner'));
      popup.appendChild(t);
      cover.style.position='fixed';
      
      // Scrolled pop-up causes unselection.
      popup.onscroll=function()
        {
        lastScroll=Date.now();
        cancelSelection();
        };
      },
    
    /**
     * Function addCover(cbCancel)
     * 
     * <p>Adds the transparent DIV cover.
     * 
     * <p>The cover is to prevent DOM events from affecting the child
     * widgets such as a list widget. Without the cover, for example,
     * child widgets may receive a click event and respond to it
     * unexpectedly when the user flicks the screen to scroll.
     * Note that only the desktop browsers need the cover.
     *
     * @type Function
     * 
     * @param {Function} cbCancel  Cancel callback function.
     * @param {Boolean}  ns        Flag for no scroll on document body.
     * 
     * @return true if cover was added.
     */
    addCover: function(cbCancel,ns)
      {
      if ( cover )
        return false;

      // Set unselectable when cover is shown.
      cover=createElement(DIV,'izrpCover');
      setSelectable(cover,false);
      setSelectable(domNode,false);
      
      // Turn off blur for Android, it's too slow due to no HW acceleration.
      //if ( navigator.userAgent.match(/Android/i) )
      //  blur=0;
 
      var self=this,b=document.body,c;
 
      if ( blur )
        {
        cover.innerHTML='<svg version="1.1" xmlns="http://www.w3.org/2000/svg"><filter id="_izrpBlur_"><feGaussianBlur stdDeviation="2"/></filter></svg>';
        c=b.className;
        c=c.replace(izrpUnblur,'');
        if ( c.indexOf(izrpBlur)<0 )
          c+=izrpBlur;
 
        b.className=c;
        }
 
      b.appendChild(cover);
      setTimeout(function()
        {
        var v=' izrpShow';
        if ( cover && b.className.indexOf(v)<0 )
          b.className+=v;
        },0);
      
      var down,ourCover=cover;
      self.register(function(e)
        {
        // If cover is "gone" for the short jiffy, eat all mouse/touch.
        var n;
        if ( cover===ourCover )
          for ( n=e.target; n; n=n.parentElement )
            if ( n==popup )
              return true;
            else
            if ( n==cover )
              {
              if ( cbCancel )
                {
                if ( self.isPressEvent(e) )
                  down=1;
                else
                if ( down && self.isReleaseEvent(e) )
                  {
                  down=0;
                  cbCancel();
                  }
                }
   
              break;
              }
        
        // Check for touch-move.
        if ( ns && e.type==touchmove )
          for ( n=e.target; n; n=n.parentNode )
            if ( n==document.body || n==cover || n==popup )
              {
              if ( e.cancelable )
                e.preventDefault();
 
              break;
              }
 
        return false;
        },ns);
      
      return true;
      },
      
    /**
     * Function removeCover(cb)
     * 
     * <p>Removes the transparent DIV cover.
     * 
     * @type Function
     * 
     * @param {Function} cb  Callback when cover has been removed, null for none.
     */
    removeCover: function(cb)
      {
      var v=cover,d=domNode;
      
      function close()
        {
        v.style.display=NONE;
        setSelectable(d,true);
        destroyElement(v);
 
        // Callback.
        if ( cb )
          cb();
        }
    
      if ( v )
        {
        // Unregister...
        this.register();
        destroyElement(popup);
        cover=popup=0;

        // Remove cover in a jiffy or now...
        if ( cb )
          setTimeout(close,100);
        else
          close();
        
        var b=document.body,
            c=b.className.replace('izrpShow','');
            
        if ( blur )
          {
          c+=izrpUnblur;
          setTimeout(function()
            {
            c=b.className;
            if ( !cover && c.indexOf(izrpUnblur)>=0 )
              b.className=c.replace(izrpBlur,'').replace(izrpUnblur,'');
            },0);
          }
        
        b.className=c;
        blur=1;
        }
      },
      
    /**
     * Function isOpen()
     * 
     * @type Function
     * 
     * @return {Boolean} true if the cover is present, thus displaying a pop-up, false otherwise.
     */
    isOpen: function()
      {
      return !!cover;
      },
   
    /**
     * Function destroy()
     * 
     * <p>Destroys the prompt dialog if visible without calling the "cbCancel" function.
     * 
     * @type Function
     */
    destroy: function()
      {
      this.removeCover();
      }
    };
  })(window,document);
