/**
 * -------------------------
 * JavaScript mixin methods.
 * -------------------------
 */

/* jshint unused:true, undef:true */
/* globals window */

(function(window) {

/**
 * Function _mixin(dest,source,copyFunc)
 * 
 * <p>Copies/adds all properties of source to a new returned <code>dest</code> Object.
 * 
 * <p>All properties, including functions (sometimes termed "methods"), excluding any non-standard extensions
 * found in Object.prototype, are copied/added to <code>dest</code>. Copying/adding each particular property is
 * delegated to <code>copyFunc</code> (if any) (<code>copyFunc</code> defaults to the Javascript assignment
 * operator if not provided).
 * 
 * <p>Notice that by default, <code>_mixin</code> executes a so-called "shallow copy" and aggregate types
 * are copied/added by reference.
 * 
 * @type Function
 * 
 * @param {Object}             dest      The object to which to copy/add all properties contained in source.  
 * @param {Object}             source    The object from which to draw all properties to copy into dest.  
 * @param {Function|undefined} copyFunc  The process used to copy/add a property in source;
 *                                       defaults to the JavaScript assignment operator.
 *                                       
 * @return {Object} The destination Object <code>dest</code> as modified.
 *                  <b>Note:</b> if <code>source</code> is falsy, nothing is done, <code>dest</code> is returned.
 */
function _mixin(dest,source,copyFunc)
  {
  var s,empty={};
  if ( source )
    for ( var name in source )
      {
      // The (!(name in empty) || empty[name] !== s) condition avoids copying properties in "source"
      // inherited from Object.prototype. For example, if dest has a custom toString() method,
      // don't overwrite it with the toString() method that source inherited from Object.prototype.
      s=source[name];
      if ( !(name in dest) || (dest[name]!==s && (!(name in empty) || empty[name]!==s)) )
        dest[name]=copyFunc? copyFunc(s): s;
      }

  return dest;
  }
  
// Save in global.
window._mixin=_mixin;

/**
 * Function mixin(dest,sources...)
 *
 * <p>Copies/adds all properties of one or more sources to <code>dest</code> Object. If <code>dest</code>
 * is falsy (e.g. null or undefined), a new Object is created and returned.
 *
 * <p>All properties, including functions (sometimes termed "methods"), excluding any non-standard extensions
 * found in <code>Object.prototype</code>, are copied/added from <code>sources</code> to <code>dest</code>.
 * <code>sources</code> are processed left to right. The JavaScript assignment operator is used to copy/add
 * each property; therefore, by default, <code>mixin</code> executes a so-called "shallow copy" and aggregate
 * types are copied/added by reference.
 *
 * @type Function
 * 
 * @param {Object} dest      The object to which to copy/add all properties contained in source.
 *                           If <code>dest</code> is falsy, then a new Object is created before
 *                           copying/adding properties begins.
 * @param {Object} sources   One of more objects from which to draw all properties to copy into
 *                           <code>dest</code>. <code>sources</code> are processed left-to-right
 *                           and if more than one of these objects contain the same property name,
 *                           the right-most value "wins".  
 *
 * @return {Object} The destination Object <code>dest</code> as modified.
 */
window.mixin=function(dest/*,sources*/)
  {
  for ( var ii=1, _dest=dest || {}; ii<arguments.length; )
    _mixin(_dest,arguments[ii++]);
  
  return _dest;
  };

// End of wrap in function.
})(window);
